# AuthDemo - Tag 3: Authentifizierung gegenüber einer Datenbank

**Java Web Aufbau - Tag 3 von 10**  
*Von Elyndra Valen, Senior Developer bei Java Fleet Systems Consulting*

---

## 📦 Projekt-Übersicht

Dieses Maven-Projekt demonstriert ein **production-ready Login-System** mit:
- ✅ Passwort-Hashing mit SHA-256
- ✅ SQL-Injection-Schutz mit Prepared Statements
- ✅ Session-basierter Authentifizierung
- ✅ JDBC Connection-Handling

---

## 🗂️ Projekt-Struktur

```
AuthDemo-Tag03/
├── src/
│   ├── main/
│   │   ├── java/
│   │   │   └── com/
│   │   │       └── ibb/
│   │   │           ├── ctrl/           # Servlets
│   │   │           │   └── SigninServlet.java
│   │   │           ├── model/          # Domain-Modelle
│   │   │           │   └── Customer.java
│   │   │           └── util/           # Helper-Klassen
│   │   │               └── GConnection.java
│   │   ├── resources/                  # Konfigurationsdateien
│   │   └── webapp/                     # Web-Ressourcen
│   │       ├── WEB-INF/
│   │       │   └── web.xml
│   │       ├── signin.jsp              # Login-Formular
│   │       ├── index.jsp               # Dashboard
│   │       └── signout.jsp             # Logout
│   └── test/                           # JUnit Tests
├── pom.xml                             # Maven Dependencies
└── README.md                           # Diese Datei
```

---

## 🔧 Voraussetzungen

**Software:**
- ✅ Java JDK 17 oder höher
- ✅ Maven 3.6 oder höher
- ✅ Payara Server 6.x oder Tomcat 10.x
- ✅ MySQL 8.0 oder höher
- ✅ NetBeans IDE (empfohlen) oder IntelliJ IDEA

---

## 🗄️ Datenbank-Setup

### 1. Datenbank erstellen
```sql
CREATE DATABASE shopdb CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE shopdb;
```

### 2. Tabelle erstellen
```sql
CREATE TABLE customer (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    email VARCHAR(255) NOT NULL UNIQUE,
    password VARCHAR(64) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

### 3. Test-User anlegen
```sql
-- Passwort: test123
INSERT INTO customer (email, password) 
VALUES ('test@example.com', 'ecd71870d1963316a97e3ac3408c9835ad8cf0f3c1bc703527c30265534f75ae');
```

**Hinweis:** Der Hash `ecd71...75ae` entspricht dem Passwort "test123" mit SHA-256.

---

## 🚀 Installation & Start

### Option 1: Mit NetBeans IDE

1. **Projekt öffnen:**
   - File → Open Project
   - `Onlineshop-Tag03-Aufbau` auswählen

2. **Dependencies laden:**
   - Rechtsklick auf Projekt → Clean and Build
   - Maven lädt automatisch alle Dependencies

3. **Server konfigurieren:**
   - Rechtsklick auf Projekt → Properties → Run
   - Server: Payara Server auswählen
   - Context Path: `/AuthDemo`

4. **Datenbank-Verbindung anpassen:**
   - `GConnection.java` öffnen
   - DB-Credentials anpassen:
   ```java
   private static final String DB_URL = "jdbc:mysql://localhost:3306/shopdb";
   private static final String DB_USER = "root";
   private static final String DB_PASSWORD = "root";
   ```

5. **Projekt deployen:**
   - F6 drücken oder Run → Run Project

6. **App öffnen:**
   - Browser: `http://localhost:8080/AuthDemo/signin.jsp`

---

### Option 2: Mit Maven CLI

```bash
# Dependencies installieren
mvn clean install

# WAR-Datei erstellen
mvn package

# Deployment auf Payara Server
# (Server muss laufen)
asadmin deploy target/AuthDemo-1.0.war
```

---

## 📝 Wichtige Dateien erklärt

### SigninServlet.java
Das Haupt-Servlet für die Authentifizierung.

**Hauptmethoden:**
- `doPost()` - Verarbeitet Login-Requests
- `find()` - Sucht User in Datenbank
- `getPassword()` - Hasht Passwörter mit SHA-256

**Security-Features:**
- ✅ Prepared Statements (SQL-Injection-Schutz)
- ✅ SHA-256 Passwort-Hashing
- ✅ Session-Management

---

### GConnection.java
Basis-Klasse für JDBC-Verbindungen.

**Funktion:**
- Stellt Connection zur Datenbank her
- Lädt MySQL JDBC Driver
- Zentrale Konfiguration

**Warum als Servlet-Parent?**
Alle Servlets erben von `GConnection` und können `getConnection()` verwenden.

---

### Customer.java
Domain-Modell für User.

**Felder:**
- `id` - Primärschlüssel
- `email` - User-Identifikation
- `password` - Gehashtes Passwort

---

### signin.jsp
Login-Formular für Benutzer.

**Features:**
- Email + Passwort-Eingabe
- HTML5-Validierung
- POST-Methode (Security)

---

### signout.jsp
Logout-Funktionalität.

**Funktion:**
- Invalidiert die Session
- Redirect zu Login

---

## 🔐 Security-Features

### 1. Passwort-Hashing (SHA-256)
```java
MessageDigest digest = MessageDigest.getInstance("SHA-256");
byte[] hash = digest.digest(password.getBytes(StandardCharsets.UTF_8));
```

**Warum?**
- Passwörter werden nie im Klartext gespeichert
- Bei DB-Leak sind Passwörter unbrauchbar

---

### 2. SQL-Injection-Schutz
```java
String sql = "SELECT * FROM customer WHERE email=? AND password=?";
PreparedStatement stmt = con.prepareStatement(sql);
stmt.setString(1, email);
stmt.setString(2, hashedPassword);
```

**Warum?**
- `?` Platzhalter verhindern Code-Injection
- Alle Eingaben werden escaped

---

### 3. Session-Management
```java
HttpSession session = request.getSession();
session.setAttribute("customer", customer);
```

**Warum?**
- Zustand wird serverseitig gespeichert
- Session-ID wird als Cookie übertragen
- Automatisches Timeout

---

## 🧪 Testen

### 1. Login testen
1. Browser: `http://localhost:8080/AuthDemo/signin.jsp`
2. Email: `test@example.com`
3. Passwort: `test123`
4. Submit → Weiterleitung zu `index.jsp`

**Erwartetes Ergebnis:**
- Session wird erstellt
- User-Objekt in Session gespeichert
- Dashboard wird angezeigt

---

### 2. Falsches Passwort testen
1. Email: `test@example.com`
2. Passwort: `wrongpassword`
3. Submit → Keine Session wird erstellt

**Erwartetes Ergebnis:**
- Login schlägt fehl
- Keine Session-Erstellung
- Keine Weiterleitung

---

### 3. SQL-Injection testen
1. Email: `test@example.com' OR '1'='1`
2. Passwort: `irrelevant`
3. Submit

**Erwartetes Ergebnis:**
- Login schlägt fehl
- Prepared Statement verhindert Injection
- Keine Sicherheitslücke

---

## 🐛 Troubleshooting

### Problem: "ClassNotFoundException: com.mysql.cj.jdbc.Driver"
**Lösung:** MySQL JDBC Driver fehlt in `pom.xml`.

```xml
<dependency>
    <groupId>com.mysql</groupId>
    <artifactId>mysql-connector-j</artifactId>
    <version>8.0.33</version>
</dependency>
```

---

### Problem: "Access denied for user 'root'@'localhost'"
**Lösung:** Datenbank-Credentials in `GConnection.java` prüfen.

```java
private static final String DB_USER = "root";  // Richtig?
private static final String DB_PASSWORD = "root";  // Richtig?
```

---

### Problem: Login funktioniert nicht
**Debug:**
1. Passwort-Hash prüfen:
```java
String hash = getPassword("test123");
System.out.println("Hash: " + hash);
```

2. Mit Datenbank vergleichen:
```sql
SELECT password FROM customer WHERE email='test@example.com';
```

3. Hashes müssen identisch sein!

---

### Problem: Session wird nicht gespeichert
**Lösung:** Session **vor** Forward erstellen:

```java
// ✅ RICHTIG
if(customer != null) {
    HttpSession session = request.getSession();  // Hier!
    session.setAttribute("customer", customer);
}
RequestDispatcher rd = request.getRequestDispatcher("index.jsp");
rd.forward(request, response);
```

---

## 📚 Weiterführende Resourcen

**Offizielle Dokumentation:**
- Jakarta Servlet API: https://jakarta.ee/specifications/servlet/
- MySQL JDBC: https://dev.mysql.com/doc/connector-j/en/
- Java Security: https://docs.oracle.com/javase/8/docs/technotes/guides/security/

**Security Best Practices:**
- OWASP Authentication Cheat Sheet: https://cheatsheetseries.owasp.org/cheatsheets/Authentication_Cheat_Sheet.html
- SQL Injection Prevention: https://cheatsheetseries.owasp.org/cheatsheets/SQL_Injection_Prevention_Cheat_Sheet.html

**Tools:**
- Online SHA-256 Generator: https://emn178.github.io/online-tools/sha256.html

---

## 🎯 Lernziele

Nach Abschluss dieses Projekts kannst du:
- ✅ Ein Login-System mit Datenbank implementieren
- ✅ Passwörter sicher mit SHA-256 hashen
- ✅ SQL-Injection mit Prepared Statements verhindern
- ✅ Sessions für Authentifizierung nutzen
- ✅ JDBC-Verbindungen in Web-Apps verwalten
- ✅ Production-ready Security-Code schreiben

---

## 👥 Credits

**Entwickelt von:**  
Elyndra Valen, Senior Developer  
Java Fleet Systems Consulting

**Kurs:**  
Java Web Aufbau - Tag 3 von 10

**Website:**  
https://www.java-developer.online

---

## 📧 Support

**Fragen? Probleme?**  
feedback@java-developer.online

**Java Fleet Systems Consulting**  
*We don't build perfect systems. We build systems that are still maintainable in five years.*

🚀 Essen-Rüttenscheid, Deutschland
