# 🛡️ Container-Managed Security Demo

**Java Web Aufbau – Tag 4 von 10**  
*Von Java Fleet Systems Consulting*

---

## 📋 Übersicht

Dieses Projekt demonstriert **Container-Managed Security** mit:

- ✅ **Form-based Authentication** (Custom Login-Formular)
- ✅ **JDBC Realm** (User aus MySQL-Datenbank)
- ✅ **Rollen-basierte Zugriffskontrolle** (admin, user)
- ✅ **SHA-256 Passwort-Hashing**
- ✅ **Konfiguration für 3 Server** (Tomcat, GlassFish, Payara)

---

## 🗂️ Projektstruktur

```
ContainerSecurityDemo/
├── pom.xml                              # Maven Build-Config
├── README.md                            # Diese Datei
├── sql/
│   └── setup-database.sql               # Datenbank-Setup
├── src/main/
│   ├── java/com/javafleet/security/
│   │   └── servlet/
│   │       ├── PublicServlet.java       # Öffentlicher Bereich
│   │       ├── UserDashboardServlet.java # Geschützt: user + admin
│   │       ├── AdminPanelServlet.java   # Geschützt: nur admin
│   │       └── LogoutServlet.java       # Session beenden
│   └── webapp/
│       ├── META-INF/
│       │   └── context.xml              # Tomcat JDBC Realm
│       ├── WEB-INF/
│       │   ├── web.xml                  # Security Constraints
│       │   └── glassfish-web.xml        # GlassFish/Payara Config
│       ├── error/
│       │   ├── 403.jsp                  # Zugriff verweigert
│       │   └── 404.jsp                  # Nicht gefunden
│       ├── index.jsp                    # Startseite
│       ├── login.jsp                    # Login-Formular
│       ├── login-error.jsp              # Login fehlgeschlagen
│       └── logout-success.jsp           # Logout-Bestätigung
```

---

## 🚀 Quick Start

### 1. Datenbank einrichten

```bash
# MySQL starten und Script ausführen
mysql -u root -p < sql/setup-database.sql
```

### 2. Projekt bauen

```bash
mvn clean package
```

### 3. WAR deployen

Kopiere `target/ContainerSecurityDemo.war` in den Server.

### 4. Testen

```
http://localhost:8080/ContainerSecurityDemo/
```

**Demo-Zugangsdaten:**
| Username | Password | Rollen |
|----------|----------|--------|
| admin    | admin123 | admin, user |
| john     | john123  | user |
| jane     | jane123  | user |

---

## 🔧 Server-Konfiguration

### Apache Tomcat 10.x

**1. MySQL Connector installieren:**

```bash
# Kopiere den MySQL Connector in Tomcat's lib-Verzeichnis
cp mysql-connector-j-8.2.0.jar $TOMCAT_HOME/lib/
```

**2. context.xml ist bereits im Projekt:**

Die Datei `src/main/webapp/META-INF/context.xml` enthält:
- DataSource-Definition
- JDBC Realm Konfiguration

**3. Server starten und testen:**

```bash
$TOMCAT_HOME/bin/startup.sh
# Browser: http://localhost:8080/ContainerSecurityDemo/
```

---

### GlassFish 7.x / Payara 6.x

**1. MySQL Connector installieren:**

```bash
cp mysql-connector-j-8.2.0.jar $GLASSFISH_HOME/glassfish/lib/
```

**2. Server starten:**

```bash
$GLASSFISH_HOME/bin/asadmin start-domain
```

**3. Admin Console öffnen:**

```
http://localhost:4848
```

**4. JDBC Connection Pool erstellen:**

```
Resources → JDBC → JDBC Connection Pools → New

General Settings:
  Pool Name: securityPool
  Resource Type: javax.sql.DataSource
  Database Driver Vendor: MySQL

Additional Properties:
  serverName: localhost
  portNumber: 3306
  databaseName: securitydb
  user: root
  password: root
  useSSL: false
```

**Ping testen!** → Muss erfolgreich sein.

**5. JDBC Resource erstellen:**

```
Resources → JDBC → JDBC Resources → New

JNDI Name: jdbc/securityDS
Pool Name: securityPool
```

**6. JDBC Realm erstellen:**

```
Configurations → server-config → Security → Realms → New

Realm Name: jdbcRealm
Class Name: com.sun.enterprise.security.auth.realm.jdbc.JDBCRealm

Properties:
  jaas-context: jdbcRealm
  datasource-jndi: jdbc/securityDS
  user-table: users
  user-name-column: username
  password-column: password
  group-table: user_roles
  group-name-column: role_name
  digest-algorithm: SHA-256
  encoding: Hex
```

**7. Server neu starten:**

```bash
$GLASSFISH_HOME/bin/asadmin restart-domain
```

**8. WAR deployen:**

```bash
$GLASSFISH_HOME/bin/asadmin deploy target/ContainerSecurityDemo.war
```

---

## 🔐 Security-Konzepte

### Security Constraints (web.xml)

```xml
<!-- Nur admin darf auf /secure/admin/* -->
<security-constraint>
    <web-resource-collection>
        <url-pattern>/secure/admin/*</url-pattern>
    </web-resource-collection>
    <auth-constraint>
        <role-name>admin</role-name>
    </auth-constraint>
</security-constraint>
```

### Form-based Authentication

```xml
<login-config>
    <auth-method>FORM</auth-method>
    <realm-name>jdbcRealm</realm-name>
    <form-login-config>
        <form-login-page>/login.jsp</form-login-page>
        <form-error-page>/login-error.jsp</form-error-page>
    </form-login-config>
</login-config>
```

### Login-Formular (Kritische Felder!)

```html
<!-- Diese Namen sind Jakarta EE Standard! -->
<form method="POST" action="j_security_check">
    <input type="text" name="j_username">
    <input type="password" name="j_password">
</form>
```

### Programmatische Sicherheit

```java
// Im Servlet
String username = request.getUserPrincipal().getName();
boolean isAdmin = request.isUserInRole("admin");
```

---

## 🐛 Troubleshooting

### Problem: "Login failed" obwohl Credentials stimmen

**Ursache:** Hash stimmt nicht überein.

**Lösung:**
```bash
# Korrekten Hash generieren
echo -n "admin123" | sha256sum
# Ausgabe: 240be518fabd2724ddb6f04eeb1da5967448d7e831c08c8fa822809f74c720a9

# In Datenbank prüfen
mysql> SELECT password FROM users WHERE username='admin';
```

### Problem: "Realm not found"

**Ursache:** JDBC Realm nicht konfiguriert.

**Lösung:**
- Tomcat: Prüfe `context.xml`
- GlassFish/Payara: Realm in Admin Console erstellen
- Server neu starten!

### Problem: "403 Forbidden" nach Login

**Ursache:** User hat nicht die erforderliche Rolle.

**Lösung:**
```sql
-- Rolle hinzufügen
INSERT INTO user_roles VALUES ('username', 'admin');
```

### Problem: "ClassNotFoundException: com.mysql.cj.jdbc.Driver"

**Ursache:** MySQL Connector fehlt.

**Lösung:**
```bash
# Tomcat
cp mysql-connector-j-8.2.0.jar $TOMCAT_HOME/lib/

# GlassFish/Payara
cp mysql-connector-j-8.2.0.jar $GLASSFISH_HOME/glassfish/lib/
```

### Problem: Logout funktioniert nicht

**Ursache:** Bei Basic Auth cached der Browser Credentials.

**Lösung:** Form-based Auth verwenden (wie in diesem Projekt).

---

## 📊 Geschützte Bereiche

| URL-Pattern | Erforderliche Rolle | Beschreibung |
|-------------|---------------------|--------------|
| `/` | Keine | Öffentliche Startseite |
| `/public` | Keine | Öffentlicher Bereich |
| `/secure/user/*` | user ODER admin | User-Dashboard |
| `/secure/admin/*` | NUR admin | Admin-Panel |
| `/logout` | Keine (aber Session nötig) | Logout |

---

## 📚 Lernziele

Nach Abschluss dieses Projekts kannst du:

- ✅ Form-based Authentication konfigurieren
- ✅ JDBC Realm mit MySQL einrichten
- ✅ Security Constraints in web.xml definieren
- ✅ Rollen-basierte Zugriffskontrolle implementieren
- ✅ Passwörter sicher mit SHA-256 hashen
- ✅ Logout korrekt implementieren
- ✅ Auf Tomcat, GlassFish und Payara deployen

---

## 👥 Credits

**Entwickelt von:**  
Java Fleet Systems Consulting

**Kurs:**  
Java Web Aufbau – Tag 4 von 10

**Website:**  
https://www.java-developer.online

---

## 📧 Support

**Fragen? Probleme?**  
feedback@java-developer.online

---

*© 2025 Java Fleet Systems Consulting*  
*We don't build perfect systems. We build systems that are still maintainable in five years.*
