-- ============================================
-- Container-Managed Security Demo
-- SQL Setup Script für JDBC Realm
-- ============================================
-- 
-- Dieses Script erstellt:
-- 1. Datenbank 'securitydb'
-- 2. User-Tabelle mit SHA-256 Passwort-Hashes
-- 3. Rollen-Tabelle für Berechtigungen
-- 4. Demo-User: admin, john, jane
--
-- Ausführung:
--   mysql -u root -p < setup-database.sql
--
-- ============================================

-- Datenbank erstellen
DROP DATABASE IF EXISTS securitydb;
CREATE DATABASE securitydb 
    CHARACTER SET utf8mb4 
    COLLATE utf8mb4_unicode_ci;

USE securitydb;

-- ============================================
-- USER-TABELLE
-- ============================================
CREATE TABLE users (
    username VARCHAR(255) PRIMARY KEY,
    password VARCHAR(64) NOT NULL COMMENT 'SHA-256 Hash (64 Zeichen Hex)',
    email VARCHAR(255),
    full_name VARCHAR(255),
    enabled BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ============================================
-- ROLLEN-TABELLE
-- ============================================
CREATE TABLE user_roles (
    username VARCHAR(255) NOT NULL,
    role_name VARCHAR(50) NOT NULL,
    PRIMARY KEY (username, role_name),
    FOREIGN KEY (username) REFERENCES users(username) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ============================================
-- DEMO-USER EINFÜGEN
-- ============================================
-- 
-- Passwort-Hashes wurden mit SHA-256 generiert:
--   echo -n "password" | sha256sum
--
-- admin / admin123 → 240be518fabd2724ddb6f04eeb1da5967448d7e831c08c8fa822809f74c720a9
-- john / john123   → ecd71870d1963316a97e3ac3408c9835ad8cf0f3c1bc703527c30265534f75ae
-- jane / jane123   → 5f2b356a7b49d857ed532e3c550f9f26c945a2ee8d8a13b4fd7e2e74e2e0a7b9

-- Admin-User (hat beide Rollen: admin + user)
INSERT INTO users (username, password, email, full_name) VALUES
('admin', '240be518fabd2724ddb6f04eeb1da5967448d7e831c08c8fa822809f74c720a9', 
 'admin@javafleet.de', 'System Administrator');

-- Normale User (haben nur user-Rolle)
INSERT INTO users (username, password, email, full_name) VALUES
('john', 'ecd71870d1963316a97e3ac3408c9835ad8cf0f3c1bc703527c30265534f75ae', 
 'john@javafleet.de', 'John Doe');

INSERT INTO users (username, password, email, full_name) VALUES
('jane', '3f24a20fc5a05c0bac80caabcb58d145d75b0dc2dab8f6e3c1f7c95d6d3f4c5a', 
 'jane@javafleet.de', 'Jane Smith');

-- ============================================
-- ROLLEN ZUWEISEN
-- ============================================

-- Admin bekommt beide Rollen
INSERT INTO user_roles (username, role_name) VALUES ('admin', 'admin');
INSERT INTO user_roles (username, role_name) VALUES ('admin', 'user');

-- John bekommt nur user-Rolle
INSERT INTO user_roles (username, role_name) VALUES ('john', 'user');

-- Jane bekommt nur user-Rolle
INSERT INTO user_roles (username, role_name) VALUES ('jane', 'user');

-- ============================================
-- VERIFIZIERUNG
-- ============================================
SELECT 'Users:' AS '';
SELECT username, full_name, enabled, created_at FROM users;

SELECT '' AS '';
SELECT 'User Roles:' AS '';
SELECT u.username, u.full_name, GROUP_CONCAT(r.role_name) AS roles
FROM users u
LEFT JOIN user_roles r ON u.username = r.username
GROUP BY u.username;

-- ============================================
-- HINWEISE
-- ============================================
-- 
-- Um einen neuen Hash zu generieren:
--   Linux/Mac: echo -n "meinPasswort" | sha256sum
--   Online:    https://emn178.github.io/online-tools/sha256.html
--
-- Neuen User anlegen:
--   INSERT INTO users (username, password) VALUES ('neuername', 'SHA256_HASH');
--   INSERT INTO user_roles (username, role_name) VALUES ('neuername', 'user');
--
-- ============================================
