package com.javafleet.security.servlet;

import jakarta.servlet.ServletException;
import jakarta.servlet.annotation.WebServlet;
import jakarta.servlet.http.HttpServlet;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;

import java.io.IOException;
import java.io.PrintWriter;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

/**
 * Admin Panel Servlet - Streng geschützter Bereich
 * 
 * Zugänglich NUR für Rolle: admin
 * 
 * Dieses Servlet demonstriert:
 * - Exklusive Admin-Berechtigung
 * - Container lehnt automatisch User ohne admin-Rolle ab
 * 
 * @author Elyndra Valen - Java Fleet Systems Consulting
 */
@WebServlet(name = "AdminPanelServlet", urlPatterns = {"/secure/admin/panel"})
public class AdminPanelServlet extends HttpServlet {
    
    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response) 
            throws ServletException, IOException {
        
        response.setContentType("text/html;charset=UTF-8");
        PrintWriter out = response.getWriter();
        
        // User-Informationen
        String username = request.getUserPrincipal().getName();
        String currentTime = LocalDateTime.now().format(DateTimeFormatter.ofPattern("dd.MM.yyyy HH:mm:ss"));
        
        // Server-Informationen
        String serverInfo = getServletContext().getServerInfo();
        String servletVersion = getServletContext().getMajorVersion() + "." + getServletContext().getMinorVersion();
        String javaVersion = System.getProperty("java.version");
        
        out.println("<!DOCTYPE html>");
        out.println("<html lang='de'>");
        out.println("<head>");
        out.println("  <meta charset='UTF-8'>");
        out.println("  <title>⚙️ Admin Panel</title>");
        out.println("  <style>");
        out.println("    body { font-family: 'Segoe UI', sans-serif; background: #1a1a2e; color: #E8E8E8; padding: 40px; margin: 0; }");
        out.println("    .container { max-width: 900px; margin: 0 auto; }");
        out.println("    header { background: linear-gradient(135deg, rgba(255,167,38,0.2), rgba(245,124,0,0.2)); padding: 30px; border-radius: 16px 16px 0 0; border: 1px solid rgba(255,167,38,0.5); border-bottom: 3px solid #FFA726; }");
        out.println("    header h1 { color: #FFA726; margin: 0 0 10px 0; }");
        out.println("    .admin-badge { display: inline-block; background: linear-gradient(135deg, #FFA726, #F57C00); color: #1a1a2e; padding: 4px 12px; border-radius: 20px; font-size: 12px; font-weight: 600; }");
        out.println("    .content { background: rgba(42,42,74,0.7); padding: 30px; border-radius: 0 0 16px 16px; }");
        out.println("    .stats-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px; margin-bottom: 30px; }");
        out.println("    .stat-card { background: rgba(26,26,46,0.8); padding: 25px; border-radius: 12px; border: 1px solid rgba(255,167,38,0.2); text-align: center; }");
        out.println("    .stat-card .icon { font-size: 36px; margin-bottom: 10px; }");
        out.println("    .stat-card h3 { color: #9E9E9E; font-size: 12px; text-transform: uppercase; margin-bottom: 8px; }");
        out.println("    .stat-card p { color: #FFA726; font-size: 14px; font-weight: 600; margin: 0; font-family: Consolas, monospace; }");
        out.println("    .warning-box { background: rgba(239,83,80,0.1); border: 1px solid rgba(239,83,80,0.3); padding: 20px; border-radius: 10px; margin-bottom: 30px; }");
        out.println("    .warning-box h3 { color: #EF5350; margin-bottom: 10px; }");
        out.println("    .security-audit { background: rgba(66,165,245,0.1); border: 1px solid rgba(66,165,245,0.3); padding: 20px; border-radius: 10px; }");
        out.println("    .security-audit h3 { color: #42A5F5; margin-bottom: 15px; }");
        out.println("    .audit-item { display: flex; justify-content: space-between; padding: 10px 0; border-bottom: 1px solid rgba(255,255,255,0.1); }");
        out.println("    .audit-item:last-child { border-bottom: none; }");
        out.println("    .audit-item .label { color: #9E9E9E; }");
        out.println("    .audit-item .value { color: #E8E8E8; font-family: Consolas, monospace; }");
        out.println("    .actions { margin-top: 30px; display: flex; gap: 15px; flex-wrap: wrap; }");
        out.println("    .btn { padding: 12px 24px; border-radius: 8px; text-decoration: none; font-weight: 600; font-size: 14px; transition: all 0.3s; }");
        out.println("    .btn-primary { background: linear-gradient(135deg, #42A5F5, #1E88E5); color: white; }");
        out.println("    .btn-danger { background: linear-gradient(135deg, #EF5350, #C62828); color: white; }");
        out.println("    .btn-secondary { background: rgba(158,158,158,0.3); color: #E8E8E8; }");
        out.println("    .btn:hover { transform: translateY(-2px); }");
        out.println("  </style>");
        out.println("</head>");
        out.println("<body>");
        out.println("  <div class='container'>");
        
        // Header
        out.println("    <header>");
        out.println("      <h1>⚙️ Admin Panel</h1>");
        out.println("      <p>Willkommen, Administrator <strong>" + username + "</strong>! <span class='admin-badge'>ADMIN</span></p>");
        out.println("    </header>");
        
        // Content
        out.println("    <div class='content'>");
        
        // Warning Box
        out.println("      <div class='warning-box'>");
        out.println("        <h3>⚠️ Nur für Administratoren!</h3>");
        out.println("        <p>Dieser Bereich ist ausschließlich für User mit der Rolle <code>admin</code> zugänglich.</p>");
        out.println("        <p>User ohne Admin-Rechte (z.B. 'john') erhalten automatisch einen <strong>403 Forbidden</strong> Fehler.</p>");
        out.println("      </div>");
        
        // Stats Grid
        out.println("      <div class='stats-grid'>");
        
        out.println("        <div class='stat-card'>");
        out.println("          <div class='icon'>🖥️</div>");
        out.println("          <h3>Server</h3>");
        out.println("          <p>" + serverInfo + "</p>");
        out.println("        </div>");
        
        out.println("        <div class='stat-card'>");
        out.println("          <div class='icon'>☕</div>");
        out.println("          <h3>Java Version</h3>");
        out.println("          <p>" + javaVersion + "</p>");
        out.println("        </div>");
        
        out.println("        <div class='stat-card'>");
        out.println("          <div class='icon'>📋</div>");
        out.println("          <h3>Servlet API</h3>");
        out.println("          <p>" + servletVersion + "</p>");
        out.println("        </div>");
        
        out.println("        <div class='stat-card'>");
        out.println("          <div class='icon'>🕐</div>");
        out.println("          <h3>Server-Zeit</h3>");
        out.println("          <p>" + currentTime + "</p>");
        out.println("        </div>");
        
        out.println("      </div>");
        
        // Security Audit
        out.println("      <div class='security-audit'>");
        out.println("        <h3>🔒 Security Audit</h3>");
        
        out.println("        <div class='audit-item'>");
        out.println("          <span class='label'>Authentifizierter User:</span>");
        out.println("          <span class='value'>" + username + "</span>");
        out.println("        </div>");
        
        out.println("        <div class='audit-item'>");
        out.println("          <span class='label'>Principal Class:</span>");
        out.println("          <span class='value'>" + request.getUserPrincipal().getClass().getSimpleName() + "</span>");
        out.println("        </div>");
        
        out.println("        <div class='audit-item'>");
        out.println("          <span class='label'>Auth Type:</span>");
        out.println("          <span class='value'>" + request.getAuthType() + "</span>");
        out.println("        </div>");
        
        out.println("        <div class='audit-item'>");
        out.println("          <span class='label'>isUserInRole(\"admin\"):</span>");
        out.println("          <span class='value' style='color: #66BB6A;'>✓ true</span>");
        out.println("        </div>");
        
        out.println("        <div class='audit-item'>");
        out.println("          <span class='label'>isUserInRole(\"user\"):</span>");
        out.println("          <span class='value'>" + (request.isUserInRole("user") ? "✓ true" : "✗ false") + "</span>");
        out.println("        </div>");
        
        out.println("        <div class='audit-item'>");
        out.println("          <span class='label'>URL-Pattern geschützt:</span>");
        out.println("          <span class='value'>/secure/admin/*</span>");
        out.println("        </div>");
        
        out.println("      </div>");
        
        // Actions
        out.println("      <div class='actions'>");
        out.println("        <a href='" + request.getContextPath() + "/' class='btn btn-secondary'>← Startseite</a>");
        out.println("        <a href='" + request.getContextPath() + "/secure/user/dashboard' class='btn btn-primary'>👤 User Dashboard</a>");
        out.println("        <a href='" + request.getContextPath() + "/logout' class='btn btn-danger'>🚪 Logout</a>");
        out.println("      </div>");
        
        out.println("    </div>");
        out.println("  </div>");
        out.println("</body>");
        out.println("</html>");
    }
}
