package com.javafleet.security.servlet;

import jakarta.servlet.ServletException;
import jakarta.servlet.annotation.WebServlet;
import jakarta.servlet.http.HttpServlet;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.servlet.http.HttpSession;

import java.io.IOException;
import java.io.PrintWriter;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

/**
 * User Dashboard Servlet - Geschützter Bereich
 * 
 * Zugänglich für Rollen: user, admin
 * 
 * Dieses Servlet demonstriert:
 * - Container-Managed Authentication
 * - Rollen-Prüfung mit isUserInRole()
 * - Session-Informationen
 * 
 * @author Elyndra Valen - Java Fleet Systems Consulting
 */
@WebServlet(name = "UserDashboardServlet", urlPatterns = {"/secure/user/dashboard"})
public class UserDashboardServlet extends HttpServlet {
    
    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response) 
            throws ServletException, IOException {
        
        response.setContentType("text/html;charset=UTF-8");
        PrintWriter out = response.getWriter();
        
        // User-Informationen vom Container abrufen
        String username = request.getUserPrincipal().getName();
        boolean isAdmin = request.isUserInRole("admin");
        boolean isUser = request.isUserInRole("user");
        
        // Session-Informationen
        HttpSession session = request.getSession(false);
        String sessionId = session != null ? session.getId().substring(0, 8) + "..." : "N/A";
        
        // Aktuelle Zeit
        String currentTime = LocalDateTime.now().format(DateTimeFormatter.ofPattern("dd.MM.yyyy HH:mm:ss"));
        
        out.println("<!DOCTYPE html>");
        out.println("<html lang='de'>");
        out.println("<head>");
        out.println("  <meta charset='UTF-8'>");
        out.println("  <title>👤 User Dashboard</title>");
        out.println("  <style>");
        out.println("    body { font-family: 'Segoe UI', sans-serif; background: #1a1a2e; color: #E8E8E8; padding: 40px; margin: 0; }");
        out.println("    .container { max-width: 800px; margin: 0 auto; }");
        out.println("    header { background: rgba(42,42,74,0.9); padding: 30px; border-radius: 16px 16px 0 0; border-bottom: 3px solid #42A5F5; }");
        out.println("    header h1 { color: #42A5F5; margin: 0 0 10px 0; }");
        out.println("    .content { background: rgba(42,42,74,0.7); padding: 30px; border-radius: 0 0 16px 16px; }");
        out.println("    .user-info { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px; margin-bottom: 30px; }");
        out.println("    .info-card { background: rgba(26,26,46,0.8); padding: 20px; border-radius: 10px; border: 1px solid rgba(255,255,255,0.1); }");
        out.println("    .info-card h3 { color: #9E9E9E; font-size: 12px; text-transform: uppercase; margin-bottom: 8px; }");
        out.println("    .info-card p { color: #E8E8E8; font-size: 18px; font-weight: 600; margin: 0; }");
        out.println("    .roles-container { display: flex; gap: 10px; flex-wrap: wrap; }");
        out.println("    .role-badge { padding: 6px 16px; border-radius: 20px; font-size: 12px; font-weight: 600; }");
        out.println("    .role-admin { background: linear-gradient(135deg, #FFA726, #F57C00); color: #1a1a2e; }");
        out.println("    .role-user { background: linear-gradient(135deg, #42A5F5, #1E88E5); color: white; }");
        out.println("    .role-inactive { background: rgba(158,158,158,0.3); color: #666; }");
        out.println("    .actions { margin-top: 30px; display: flex; gap: 15px; flex-wrap: wrap; }");
        out.println("    .btn { padding: 12px 24px; border-radius: 8px; text-decoration: none; font-weight: 600; font-size: 14px; transition: all 0.3s; }");
        out.println("    .btn-primary { background: linear-gradient(135deg, #42A5F5, #1E88E5); color: white; }");
        out.println("    .btn-warning { background: linear-gradient(135deg, #FFA726, #F57C00); color: #1a1a2e; }");
        out.println("    .btn-danger { background: linear-gradient(135deg, #EF5350, #C62828); color: white; }");
        out.println("    .btn-secondary { background: rgba(158,158,158,0.3); color: #E8E8E8; }");
        out.println("    .btn:hover { transform: translateY(-2px); }");
        out.println("    .security-info { background: rgba(102,187,106,0.1); border: 1px solid rgba(102,187,106,0.3); padding: 20px; border-radius: 10px; margin-top: 30px; }");
        out.println("    .security-info h3 { color: #66BB6A; margin-bottom: 15px; }");
        out.println("    .security-info code { background: rgba(0,0,0,0.3); padding: 2px 8px; border-radius: 4px; font-size: 13px; }");
        out.println("  </style>");
        out.println("</head>");
        out.println("<body>");
        out.println("  <div class='container'>");
        
        // Header
        out.println("    <header>");
        out.println("      <h1>👤 User Dashboard</h1>");
        out.println("      <p>Willkommen zurück, <strong>" + username + "</strong>!</p>");
        out.println("    </header>");
        
        // Content
        out.println("    <div class='content'>");
        
        // User Info Cards
        out.println("      <div class='user-info'>");
        
        out.println("        <div class='info-card'>");
        out.println("          <h3>👤 Username</h3>");
        out.println("          <p>" + username + "</p>");
        out.println("        </div>");
        
        out.println("        <div class='info-card'>");
        out.println("          <h3>🎭 Deine Rollen</h3>");
        out.println("          <div class='roles-container'>");
        out.println("            <span class='role-badge " + (isAdmin ? "role-admin" : "role-inactive") + "'>admin</span>");
        out.println("            <span class='role-badge " + (isUser ? "role-user" : "role-inactive") + "'>user</span>");
        out.println("          </div>");
        out.println("        </div>");
        
        out.println("        <div class='info-card'>");
        out.println("          <h3>🔑 Session ID</h3>");
        out.println("          <p style='font-family: Consolas, monospace; font-size: 14px;'>" + sessionId + "</p>");
        out.println("        </div>");
        
        out.println("        <div class='info-card'>");
        out.println("          <h3>🕐 Server-Zeit</h3>");
        out.println("          <p style='font-size: 14px;'>" + currentTime + "</p>");
        out.println("        </div>");
        
        out.println("      </div>");
        
        // Actions
        out.println("      <div class='actions'>");
        out.println("        <a href='" + request.getContextPath() + "/' class='btn btn-secondary'>← Startseite</a>");
        
        if (isAdmin) {
            out.println("        <a href='" + request.getContextPath() + "/secure/admin/panel' class='btn btn-warning'>⚙️ Admin-Panel</a>");
        }
        
        out.println("        <a href='" + request.getContextPath() + "/logout' class='btn btn-danger'>🚪 Logout</a>");
        out.println("      </div>");
        
        // Security Info
        out.println("      <div class='security-info'>");
        out.println("        <h3>🔒 So funktioniert's:</h3>");
        out.println("        <p>Dieses Servlet ist durch <code>security-constraint</code> geschützt:</p>");
        out.println("        <ul>");
        out.println("          <li>URL-Pattern: <code>/secure/user/*</code></li>");
        out.println("          <li>Erlaubte Rollen: <code>admin</code>, <code>user</code></li>");
        out.println("          <li>Auth-Method: <code>FORM</code></li>");
        out.println("          <li>Realm: <code>jdbcRealm</code></li>");
        out.println("        </ul>");
        out.println("        <p style='margin-top: 15px;'>Der Container hat automatisch geprüft:</p>");
        out.println("        <ol>");
        out.println("          <li><code>request.getUserPrincipal()</code> → " + username + "</li>");
        out.println("          <li><code>request.isUserInRole(\"user\")</code> → " + isUser + "</li>");
        out.println("          <li><code>request.isUserInRole(\"admin\")</code> → " + isAdmin + "</li>");
        out.println("        </ol>");
        out.println("      </div>");
        
        out.println("    </div>");
        out.println("  </div>");
        out.println("</body>");
        out.println("</html>");
    }
}
