package com.javafleet.tags;

import jakarta.servlet.jsp.tagext.SimpleTagSupport;
import jakarta.servlet.jsp.JspException;
import java.io.IOException;

/**
 * LEVEL 2: Tag mit Attributen
 * 
 * Erzeugt Bootstrap-style Alert-Boxen mit verschiedenen Typen.
 * Demonstriert Attribute als Properties mit Getter/Setter.
 * 
 * Verwendung in JSP:
 * <fleet:alert type="success" message="Operation completed!" />
 * <fleet:alert type="danger" message="Error occurred!" />
 * <fleet:alert type="warning" message="Warning!" />
 * <fleet:alert type="info" message="FYI..." />
 * 
 * @author Elyndra Valen - Java Fleet Systems Consulting
 */
public class AlertTag extends SimpleTagSupport {
    
    // Attribute als private Felder
    private String type = "info";  // Default-Wert
    private String message;
    private boolean dismissible = false;
    
    // ========================================
    // GETTER UND SETTER - PFLICHT!
    // Container ruft Setter automatisch auf
    // ========================================
    
    public String getType() {
        return type;
    }
    
    public void setType(String type) {
        this.type = type;
    }
    
    public String getMessage() {
        return message;
    }
    
    public void setMessage(String message) {
        this.message = message;
    }
    
    public boolean isDismissible() {
        return dismissible;
    }
    
    public void setDismissible(boolean dismissible) {
        this.dismissible = dismissible;
    }
    
    @Override
    public void doTag() throws JspException, IOException {
        // Validierung
        if (message == null || message.isEmpty()) {
            throw new JspException("Attribut 'message' ist erforderlich!");
        }
        
        // Icon basierend auf Type (Java 17 Switch Expression)
        String icon = switch (type.toLowerCase()) {
            case "success" -> "✓";
            case "danger", "error" -> "✗";
            case "warning" -> "⚠";
            default -> "ℹ";
        };
        
        // CSS-Klasse für Type
        String cssType = type.toLowerCase();
        if ("error".equals(cssType)) {
            cssType = "danger";
        }
        
        // Dismissible Button
        String dismissBtn = dismissible 
            ? "<button type='button' class='close' onclick='this.parentElement.style.display=\"none\"'>&times;</button>" 
            : "";
        
        // HTML generieren
        String html = String.format("""
            <div class="alert alert-%s%s" role="alert">
                %s
                <span class="alert-icon">%s</span>
                <span class="alert-message">%s</span>
            </div>
            """, 
            cssType,
            dismissible ? " alert-dismissible" : "",
            dismissBtn,
            icon, 
            escapeHtml(message)
        );
        
        getJspContext().getOut().write(html);
    }
    
    /**
     * Einfaches HTML-Escaping für Sicherheit
     */
    private String escapeHtml(String input) {
        if (input == null) return "";
        return input
            .replace("&", "&amp;")
            .replace("<", "&lt;")
            .replace(">", "&gt;")
            .replace("\"", "&quot;")
            .replace("'", "&#39;");
    }
}
