package com.javafleet.tags;

import jakarta.servlet.jsp.tagext.SimpleTagSupport;
import jakarta.servlet.jsp.JspException;
import java.io.IOException;
import java.time.LocalTime;

/**
 * LEVEL 2: Greeting Tag
 * 
 * Zeigt personalisierte Grüße basierend auf Tageszeit.
 * Kann automatisch (Server-Zeit) oder manuell (time-Attribut) funktionieren.
 * 
 * Verwendung in JSP:
 * <fleet:greeting name="Nova" />
 * <fleet:greeting name="${user.name}" time="morning" />
 * <fleet:greeting name="Admin" time="evening" lang="de" />
 * 
 * @author Elyndra Valen - Java Fleet Systems Consulting
 */
public class GreetingTag extends SimpleTagSupport {
    
    private String name;
    private String time;  // morning, afternoon, evening oder null (auto)
    private String lang = "en";  // en oder de
    
    public String getName() {
        return name;
    }
    
    public void setName(String name) {
        this.name = name;
    }
    
    public String getTime() {
        return time;
    }
    
    public void setTime(String time) {
        this.time = time;
    }
    
    public String getLang() {
        return lang;
    }
    
    public void setLang(String lang) {
        this.lang = lang;
    }
    
    @Override
    public void doTag() throws JspException, IOException {
        if (name == null || name.isEmpty()) {
            name = "Guest";
        }
        
        // Tageszeit ermitteln
        String timeOfDay = time;
        if (timeOfDay == null || timeOfDay.isEmpty()) {
            timeOfDay = determineTimeOfDay();
        }
        
        // Begrüßung basierend auf Sprache und Tageszeit
        String greeting = getGreeting(timeOfDay, lang);
        
        String html = String.format(
            "<span class=\"greeting greeting-%s\">%s, %s!</span>",
            timeOfDay,
            greeting,
            escapeHtml(name)
        );
        
        getJspContext().getOut().write(html);
    }
    
    private String determineTimeOfDay() {
        int hour = LocalTime.now().getHour();
        if (hour >= 5 && hour < 12) {
            return "morning";
        } else if (hour >= 12 && hour < 17) {
            return "afternoon";
        } else if (hour >= 17 && hour < 21) {
            return "evening";
        } else {
            return "night";
        }
    }
    
    private String getGreeting(String timeOfDay, String language) {
        return switch (language.toLowerCase()) {
            case "de" -> switch (timeOfDay) {
                case "morning" -> "Guten Morgen";
                case "afternoon" -> "Guten Tag";
                case "evening" -> "Guten Abend";
                case "night" -> "Gute Nacht";
                default -> "Hallo";
            };
            default -> switch (timeOfDay) {
                case "morning" -> "Good morning";
                case "afternoon" -> "Good afternoon";
                case "evening" -> "Good evening";
                case "night" -> "Good night";
                default -> "Hello";
            };
        };
    }
    
    private String escapeHtml(String input) {
        if (input == null) return "";
        return input
            .replace("&", "&amp;")
            .replace("<", "&lt;")
            .replace(">", "&gt;");
    }
}
