# Java Web Aufbau - Tag 7

## JPA vs JDBC - Konfiguration & Provider

**Von Elyndra Valen** | Java Fleet Systems Consulting

Vollständiges Maven-Projekt mit allen Code-Beispielen aus dem Tutorial.

---

## 📋 Übersicht

Dieses Projekt demonstriert alle JPA-Konzepte aus **Tag 7: JPA vs JDBC - Konfiguration & Provider**.

### Was du lernst:

- ✅ **Entity-Klassen** mit `@Entity`, `@Id`, `@Column`
- ✅ **EntityManager** für CRUD-Operationen
- ✅ **persistence.xml** Konfiguration (Dev & Production)
- ✅ **Named Queries** und **JPQL**
- ✅ **Dirty Checking** in der Praxis
- ✅ **@PrePersist** Lifecycle Callbacks
- ✅ **DataSource** über JNDI
- ✅ **Transaction Management** mit `@Stateless`

---

## 🚀 Quick Start

### 1. Voraussetzungen

- **JDK 21 LTS** installiert
- **Apache Maven 3.8+**
- **Payara Server 6.x** (oder GlassFish)
- **MySQL 8.0+** (via Docker oder lokal)

### 2. Datenbank starten (Docker)

```bash
docker run --name mysql-jpa \
  -e MYSQL_ROOT_PASSWORD=secret \
  -e MYSQL_DATABASE=jpadb \
  -p 3306:3306 \
  -d mysql:8
```

**Datenbank-Details:**
- Host: `localhost:3306`
- Database: `jpadb`
- User: `root`
- Password: `secret`

### 3. DataSource in Payara konfigurieren

#### Schritt 1: JDBC Connection Pool

1. Öffne **Payara Admin Console**: http://localhost:4848
2. Navigiere zu: **Resources → JDBC → JDBC Connection Pools**
3. Klicke auf **New...**
4. Konfiguration:
   - **Pool Name**: `MySQLPool`
   - **Resource Type**: `javax.sql.DataSource`
   - **Database Driver Vendor**: `MySQL`
5. **Next**, dann folgende Properties setzen:
   - `serverName`: `localhost`
   - `portNumber`: `3306`
   - `databaseName`: `jpadb`
   - `user`: `root`
   - `password`: `secret`
6. **Finish**
7. **Ping** testen!

#### Schritt 2: JDBC Resource

1. Navigiere zu: **Resources → JDBC → JDBC Resources**
2. Klicke auf **New...**
3. Konfiguration:
   - **JNDI Name**: `jdbc/myDB`
   - **Pool Name**: `MySQLPool`
4. **OK**

### 4. Projekt bauen & deployen

```bash
# Projekt bauen
mvn clean package

# WAR-Datei wird erstellt: target/jpa-beispiele-tag7.war
```

**Deployment:**
1. Payara Admin Console → **Applications**
2. **Deploy...**
3. WAR-Datei auswählen: `target/jpa-beispiele-tag7.war`
4. **OK**

### 5. Anwendung öffnen

```
http://localhost:8080/jpa-beispiele-tag7/
```

---

## 📁 Projekt-Struktur

```
JavaWebAufbau-Tag7/
├── pom.xml                                 # Maven Konfiguration
├── README.md                               # Diese Datei
└── src/
    └── main/
        ├── java/com/javafleet/
        │   ├── model/
        │   │   ├── User.java              # User Entity (Grundlagen)
        │   │   └── Product.java           # Product Entity (Challenge)
        │   ├── service/
        │   │   ├── UserService.java       # User CRUD Service
        │   │   └── ProductService.java    # Product CRUD Service
        │   └── servlet/
        │       ├── UserServlet.java       # User Web-Controller
        │       └── ProductServlet.java    # Product Web-Controller
        ├── resources/
        │   └── META-INF/
        │       └── persistence.xml        # JPA Konfiguration
        └── webapp/
            ├── index.jsp                   # Startseite
            ├── css/
            │   └── style.css              # Dark Orange/White Theme
            └── WEB-INF/
                ├── web.xml                 # Deployment Descriptor
                └── views/
                    ├── users.jsp          # User Management View
                    └── products.jsp       # Product Management View
```

---

## 🎯 Features

### 1. User Management

**Entity:** `User.java`
- Auto-generated ID (`@GeneratedValue`)
- Username (unique, NOT NULL)
- Email
- CreatedAt Timestamp (`@PrePersist`)

**Service:** `UserService.java`
- `createUser()` - persist()
- `findUser()` - find()
- `updateEmail()` - Dirty Checking
- `deleteUser()` - remove()
- `findUsersByEmail()` - JPQL Query
- `findByEmailExact()` - Named Query

**Web-Interface:** `/users`
- User erstellen
- User suchen
- User löschen
- Alle User anzeigen

### 2. Product Management (Challenge Lösung)

**Entity:** `Product.java`
- Auto-generated ID
- Name (max 100 Zeichen)
- Price (BigDecimal)
- Stock (Integer, default 0)
- CreatedAt Timestamp

**Service:** `ProductService.java`
- `createProduct()`
- `findProductById()`
- `findAllProducts()` - Named Query
- `updateStock()` - Dirty Checking
- `increaseStock()` / `decreaseStock()`
- `deleteProduct()`
- `findLowStockProducts()` - Custom JPQL

**Web-Interface:** `/products`
- Product erstellen
- Stock aktualisieren
- Products löschen
- Low-Stock Filter

---

## 📚 Code-Beispiele

### Entity erstellen

```java
@Entity
@Table(name = "users")
public class User {
    
    @Id
    @GeneratedValue(strategy = GenerationType.IDENTITY)
    private Long id;
    
    @Column(name = "username", nullable = false, unique = true)
    private String username;
    
    @Column(name = "email", nullable = false)
    private String email;
    
    @PrePersist
    protected void onCreate() {
        createdAt = LocalDateTime.now();
    }
    
    // Getters & Setters...
}
```

### CRUD mit EntityManager

```java
@Stateless
public class UserService {
    
    @PersistenceContext
    private EntityManager em;
    
    // CREATE
    public void createUser(String username, String email) {
        User user = new User(username, email);
        em.persist(user);  // INSERT beim Commit
    }
    
    // READ
    public User findUser(Long id) {
        return em.find(User.class, id);
    }
    
    // UPDATE (Dirty Checking!)
    public void updateEmail(Long id, String newEmail) {
        User user = em.find(User.class, id);
        if (user != null) {
            user.setEmail(newEmail);
            // Automatisches UPDATE beim Commit!
        }
    }
    
    // DELETE
    public void deleteUser(Long id) {
        User user = em.find(User.class, id);
        if (user != null) {
            em.remove(user);
        }
    }
}
```

### JPQL Query

```java
public List<User> findUsersByEmail(String pattern) {
    return em.createQuery(
        "SELECT u FROM User u WHERE u.email LIKE :pattern", 
        User.class)
        .setParameter("pattern", "%" + pattern + "%")
        .getResultList();
}
```

### Named Query

```java
// In Entity:
@NamedQuery(
    name = "Product.findAll",
    query = "SELECT p FROM Product p ORDER BY p.name"
)

// In Service:
public List<Product> findAllProducts() {
    return em.createNamedQuery("Product.findAll", Product.class)
        .getResultList();
}
```

---

## ⚙️ Konfiguration

### persistence.xml

**Development Mode:**
```xml
<persistence-unit name="myPU" transaction-type="JTA">
    <jta-data-source>jdbc/myDB</jta-data-source>
    <properties>
        <property name="hibernate.dialect" 
                  value="org.hibernate.dialect.MySQL8Dialect"/>
        <property name="hibernate.show_sql" value="true"/>
        <property name="hibernate.hbm2ddl.auto" value="update"/>
    </properties>
</persistence-unit>
```

**Production Mode:**
```xml
<persistence-unit name="myPU-prod" transaction-type="JTA">
    <jta-data-source>jdbc/myDB</jta-data-source>
    <properties>
        <property name="hibernate.dialect" 
                  value="org.hibernate.dialect.MySQL8Dialect"/>
        <property name="hibernate.show_sql" value="false"/>
        <!-- WICHTIG: In Production NIEMALS "update"! -->
        <property name="hibernate.hbm2ddl.auto" value="none"/>
    </properties>
</persistence-unit>
```

---

## 🔧 Troubleshooting

### Problem: `LazyInitializationException`

**Ursache:** LAZY Collection außerhalb Transaction geladen.

**Lösung:**
```java
// Option 1: JOIN FETCH
em.createQuery(
    "SELECT u FROM User u LEFT JOIN FETCH u.orders WHERE u.id = :id", 
    User.class)
    .setParameter("id", id)
    .getSingleResult();

// Option 2: FetchType.EAGER (nicht empfohlen!)
@OneToMany(fetch = FetchType.EAGER)
```

### Problem: `No Persistence provider`

**Ursache:** Hibernate nicht gefunden oder `persistence.xml` falsch platziert.

**Lösung:**
1. Prüfe: `persistence.xml` liegt in `src/main/resources/META-INF/`
2. Füge Provider hinzu:
```xml
<provider>org.hibernate.jpa.HibernatePersistenceProvider</provider>
```

### Problem: `Table doesn't exist`

**Ursache:** Schema nicht erstellt.

**Lösung Development:**
```xml
<property name="hibernate.hbm2ddl.auto" value="update"/>
```

**Lösung Production:**
Erstelle Tabellen manuell oder nutze Flyway/Liquibase!

```sql
CREATE TABLE users (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) NOT NULL UNIQUE,
    email VARCHAR(100) NOT NULL,
    created_at DATETIME
);

CREATE TABLE products (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    price DECIMAL(10,2) NOT NULL,
    stock INT NOT NULL DEFAULT 0,
    created_at DATETIME
);
```

---

## 📖 Ressourcen

**Original Tutorial:**
- https://www.java-developer.online/tag7-jpa-vs-jdbc-konfiguration-provider/

**Offizielle Dokumentation:**
- Jakarta Persistence: https://jakarta.ee/specifications/persistence/
- Hibernate ORM: https://hibernate.org/orm/documentation/

**Best Practices:**
- Vlad Mihalcea: "High-Performance Java Persistence"
- Thorben Janssen: https://thorben-janssen.com/tips/

---

## 🎓 Lernziele

Nach diesem Projekt beherrschst du:

- [x] Den Unterschied zwischen JDBC und JPA
- [x] `persistence.xml` für Dev & Production konfigurieren
- [x] Entity-Klassen mit korrekten Annotations erstellen
- [x] EntityManager für alle CRUD-Operationen nutzen
- [x] JPQL und Named Queries schreiben
- [x] Dirty Checking verstehen und anwenden
- [x] DataSource über JNDI einbinden
- [x] Transaction Management mit `@Stateless`

---

## 💬 Feedback

War dieses Projekt hilfreich? Feedback willkommen!

**Email:** feedback@java-developer.online

---

## 📄 Lizenz

© 2025 Java Fleet Systems Consulting

Tutorial & Projekt für Lernzwecke - Viel Erfolg! 🚀

---

**Java Web Aufbau - Tag 7 von 10**

Nächster Schritt: **Tag 8 - JPA Relationen (1): @OneToOne & @ManyToOne**
