package com.javafleet.config;

import com.javafleet.service.ProductService;
import com.javafleet.service.UserService;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.Resource;
import jakarta.ejb.EJB;
import jakarta.ejb.Singleton;
import jakarta.ejb.Startup;
import jakarta.ejb.Timeout;
import jakarta.ejb.Timer;
import jakarta.ejb.TimerConfig;
import jakarta.ejb.TimerService;
import java.math.BigDecimal;
import java.util.logging.Logger;

/**
 * DatabaseSeeder - Initialisiert Beispieldaten beim Anwendungsstart
 *
 * Annotations:
 * - @Singleton: Nur eine Instanz dieser Klasse wird erstellt
 * - @Startup: Wird automatisch beim Server-Start initialisiert
 * - @PostConstruct: Methode wird nach der Initialisierung aufgerufen
 *
 * Features:
 * - Prüft ob Daten vorhanden sind
 * - Fügt nur Daten ein, wenn DB leer ist
 * - Loggt alle Aktionen
 * - Verwendet Timer für verzögerte Initialisierung (Web-Container muss bereit sein)
 */
@Singleton
@Startup
public class DatabaseSeeder {

    private static final Logger LOGGER = Logger.getLogger(DatabaseSeeder.class.getName());

    @Resource
    private TimerService timerService;

    @EJB
    private UserService userService;

    @EJB
    private ProductService productService;

    /**
     * Wird automatisch nach der Bean-Initialisierung aufgerufen.
     * Startet einen Timer, der nach 2 Sekunden das Seeding auslöst,
     * damit der Web-Container vollständig gestartet ist.
     */
    @PostConstruct
    public void init() {
        LOGGER.info("=== DatabaseSeeder: Timer wird gestartet (2 Sekunden Verzögerung) ===");
        // Timer nach 2 Sekunden auslösen - gibt dem Container Zeit zum vollständigen Start
        timerService.createSingleActionTimer(2000, new TimerConfig("DatabaseSeeder", false));
    }

    /**
     * Wird vom Timer aufgerufen, wenn der Container bereit ist
     */
    @Timeout
    public void executeSeeding(Timer timer) {
        LOGGER.info("=== DatabaseSeeder wird gestartet ===");

        try {
            seedUsers();
            seedProducts();
            LOGGER.info("=== DatabaseSeeder erfolgreich abgeschlossen ===");
        } catch (Exception e) {
            LOGGER.severe("Fehler beim Seeding: " + e.getMessage());
            e.printStackTrace();
        }
    }

    /**
     * User-Beispieldaten einfügen
     */
    private void seedUsers() {
        long userCount = userService.countUsers();

        if (userCount > 0) {
            LOGGER.info("User-Tabelle enthält bereits " + userCount + " Einträge. Seeding übersprungen.");
            return;
        }

        LOGGER.info("Füge User-Beispieldaten ein...");

        // Beispiel-User einfügen
        userService.createUser("elyndra_v", "elyndra@java-fleet.com");
        userService.createUser("nova_t", "nova@java-fleet.com");
        userService.createUser("cassian_h", "cassian@java-fleet.com");
        userService.createUser("zara_k", "zara@java-fleet.com");
        userService.createUser("phoenix_r", "phoenix@java-fleet.com");
        userService.createUser("orion_m", "orion@java-fleet.com");
        userService.createUser("luna_s", "luna@java-fleet.com");
        userService.createUser("atlas_w", "atlas@java-fleet.com");

        LOGGER.info("8 User erfolgreich eingefügt.");
    }

    /**
     * Product-Beispieldaten einfügen
     */
    private void seedProducts() {
        long productCount = productService.findAllProducts().size();

        if (productCount > 0) {
            LOGGER.info("Product-Tabelle enthält bereits " + productCount + " Einträge. Seeding übersprungen.");
            return;
        }

        LOGGER.info("Füge Product-Beispieldaten ein...");

        // Java-Bücher
        productService.createProduct("Java for Dummies", new BigDecimal("29.99"), 15);
        productService.createProduct("Effective Java (3rd Edition)", new BigDecimal("44.99"), 8);
        productService.createProduct("Clean Code", new BigDecimal("39.99"), 3);
        productService.createProduct("Head First Design Patterns", new BigDecimal("34.99"), 12);
        productService.createProduct("Spring Boot in Action", new BigDecimal("49.99"), 25);

        // Programmiersprachen
        productService.createProduct("Python Crash Course", new BigDecimal("34.99"), 20);
        productService.createProduct("JavaScript: The Good Parts", new BigDecimal("27.99"), 7);
        productService.createProduct("Learning TypeScript", new BigDecimal("42.50"), 10);

        // Architektur & Design
        productService.createProduct("Domain-Driven Design", new BigDecimal("54.99"), 5);
        productService.createProduct("Refactoring (2nd Edition)", new BigDecimal("47.99"), 9);
        productService.createProduct("The Pragmatic Programmer", new BigDecimal("39.95"), 14);
        productService.createProduct("Code Complete", new BigDecimal("49.50"), 6);

        // DevOps & Tools
        productService.createProduct("Docker Deep Dive", new BigDecimal("35.99"), 18);
        productService.createProduct("Kubernetes in Action", new BigDecimal("52.99"), 11);
        productService.createProduct("Git Pocket Guide", new BigDecimal("19.99"), 30);

        // Datenbanken
        productService.createProduct("SQL Performance Explained", new BigDecimal("32.50"), 8);
        productService.createProduct("MongoDB: The Definitive Guide", new BigDecimal("44.50"), 4);
        productService.createProduct("Redis in Action", new BigDecimal("38.99"), 13);

        // Soft Skills & Karriere
        productService.createProduct("The Clean Coder", new BigDecimal("29.95"), 22);
        productService.createProduct("Cracking the Coding Interview", new BigDecimal("39.99"), 16);

        // Out of Stock Beispiele (für Test-Szenarien)
        productService.createProduct("Rare Collector's Edition: SICP", new BigDecimal("99.99"), 0);
        productService.createProduct("Limited: Algorithms 4th Edition", new BigDecimal("64.99"), 0);

        LOGGER.info("22 Products erfolgreich eingefügt.");
    }
}
