package com.javafleet.service;

import com.javafleet.model.Product;
import jakarta.ejb.Stateless;
import jakarta.inject.Named;
import jakarta.persistence.EntityManager;
import jakarta.persistence.PersistenceContext;
import jakarta.transaction.Transactional;
import java.math.BigDecimal;
import java.util.List;

/**
 * ProductService - Challenge Lösung
 * 
 * Requirements:
 * - createProduct()
 * - findProductById()
 * - findAllProducts()
 * - updateStock()
 * - deleteProduct()
 */
//@Named
//@Transactional
@Stateless
public class ProductService {
    
    @PersistenceContext(unitName = "myPU")
    private EntityManager em;
    
    /**
     * CREATE - Product erstellen
     */
    public Product createProduct(String name, BigDecimal price) {
        Product product = new Product(name, price);
        em.persist(product);
        return product;
    }
    
    public Product createProduct(String name, BigDecimal price, Integer stock) {
        Product product = new Product(name, price, stock);
        em.persist(product);
        return product;
    }
    
    /**
     * READ - Product per ID finden
     */
    public Product findProductById(Long id) {
        return em.find(Product.class, id);
    }
    
    /**
     * READ - Alle Products laden
     * Nutzt Named Query aus der Entity
     */
    public List<Product> findAllProducts() {
        return em.createNamedQuery("Product.findAll", Product.class)
            .getResultList();
    }
    
    /**
     * UPDATE - Stock aktualisieren
     * Nutzt Dirty Checking - kein merge() nötig!
     */
    public void updateStock(Long id, Integer newStock) {
        Product product = em.find(Product.class, id);
        if (product != null) {
            product.setStock(newStock);
            // Automatisches UPDATE beim Commit!
        }
    }
    
    /**
     * UPDATE - Stock erhöhen (z.B. Wareneingang)
     */
    public void increaseStock(Long id, Integer amount) {
        Product product = em.find(Product.class, id);
        if (product != null) {
            product.setStock(product.getStock() + amount);
        }
    }
    
    /**
     * UPDATE - Stock verringern (z.B. Verkauf)
     * Returns true wenn erfolgreich, false wenn nicht genug Stock
     */
    public boolean decreaseStock(Long id, Integer amount) {
        Product product = em.find(Product.class, id);
        if (product != null && product.getStock() >= amount) {
            product.setStock(product.getStock() - amount);
            return true;
        }
        return false;
    }
    
    /**
     * DELETE - Product löschen
     */
    public void deleteProduct(Long id) {
        Product product = em.find(Product.class, id);
        if (product != null) {
            em.remove(product);
        }
    }
    
    /**
     * Alle Products unter einem bestimmten Preis
     */
    public List<Product> findProductsUnderPrice(BigDecimal maxPrice) {
        return em.createQuery(
                "SELECT p FROM Product p WHERE p.price <= :maxPrice ORDER BY p.price", 
                Product.class
            )
            .setParameter("maxPrice", maxPrice)
            .getResultList();
    }
    
    /**
     * Products mit niedrigem Stock finden
     */
    public List<Product> findLowStockProducts(Integer threshold) {
        return em.createQuery(
                "SELECT p FROM Product p WHERE p.stock <= :threshold ORDER BY p.stock", 
                Product.class
            )
            .setParameter("threshold", threshold)
            .getResultList();
    }
}
