package com.javafleet.service;

import com.javafleet.model.User;
import jakarta.ejb.Stateless;
import jakarta.persistence.EntityManager;
import jakarta.persistence.PersistenceContext;
import java.util.List;

/**
 * UserService - Alle EntityManager CRUD Operationen
 * 
 * Zeigt:
 * - @Stateless für automatisches Transaction Management
 * - @PersistenceContext für EntityManager Injection
 * - persist(), find(), merge(), remove()
 * - JPQL Queries mit createQuery()
 * - Named Queries
 * - Dirty Checking (automatisches UPDATE)
 */
@Stateless
public class UserService {
    
    @PersistenceContext(unitName = "myPU")
    private EntityManager em;
    
    /**
     * CREATE - User erstellen
     * persist() macht INSERT beim Transaction-Commit
     */
    public void createUser(String username, String email) {
        User user = new User(username, email);
        em.persist(user);  // INSERT beim Transaction-Commit!
    }
    
    /**
     * READ - User per ID finden
     * find() macht SELECT per Primary Key
     */
    public User findUser(Long id) {
        return em.find(User.class, id);
    }
    
    /**
     * UPDATE - Email ändern
     * Keine merge() nötig! Dirty Checking macht automatisches UPDATE
     */
    public void updateEmail(Long id, String newEmail) {
        User user = em.find(User.class, id);
        if (user != null) {
            user.setEmail(newEmail);
            // Automatisches UPDATE dank Dirty Checking!
            // JPA sieht die Änderung und generiert UPDATE beim Commit
        }
    }
    
    /**
     * DELETE - User löschen
     * remove() macht DELETE beim Transaction-Commit
     */
    public void deleteUser(Long id) {
        User user = em.find(User.class, id);
        if (user != null) {
            em.remove(user);
        }
    }
    
    /**
     * QUERY - User per Email-Pattern suchen
     * JPQL Query mit Parameter-Binding
     */
    public List<User> findUsersByEmail(String pattern) {
        return em.createQuery(
                "SELECT u FROM User u WHERE u.email LIKE :pattern", 
                User.class
            )
            .setParameter("pattern", "%" + pattern + "%")
            .getResultList();
    }
    
    /**
     * Named Query nutzen
     * Query ist in @NamedQuery der Entity definiert
     */
    public User findByEmailExact(String email) {
        try {
            return em.createNamedQuery("User.findByEmail", User.class)
                .setParameter("email", email)
                .getSingleResult();
        } catch (Exception e) {
            return null;  // User nicht gefunden
        }
    }
    
    /**
     * Alle User laden
     */
    public List<User> findAllUsers() {
        return em.createQuery("SELECT u FROM User u ORDER BY u.username", User.class)
            .getResultList();
    }
    
    /**
     * User zählen
     */
    public long countUsers() {
        return em.createQuery("SELECT COUNT(u) FROM User u", Long.class)
            .getSingleResult();
    }
}
