package com.javafleet.servlet;

import com.javafleet.model.Product;
import com.javafleet.service.ProductService;
import jakarta.inject.Inject;
import jakarta.servlet.ServletException;
import jakarta.servlet.annotation.WebServlet;
import jakarta.servlet.http.HttpServlet;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.List;

/**
 * ProductServlet - Web-Interface für Product CRUD (Challenge Lösung)
 * 
 * Actions:
 * - list: Alle Products anzeigen
 * - create: Product erstellen
 * - updateStock: Stock ändern
 * - delete: Product löschen
 */
@WebServlet("/products")
public class ProductServlet extends HttpServlet {
    
    @Inject
    private ProductService productService;
    
    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response) 
            throws ServletException, IOException {
        
        String action = request.getParameter("action");
        
        if ("delete".equals(action)) {
            handleDelete(request, response);
            return;
        }
        
        if ("lowstock".equals(action)) {
            handleLowStock(request, response);
            return;
        }
        
        // Default: Liste anzeigen
        List<Product> products = productService.findAllProducts();
        
        request.setAttribute("products", products);
        request.getRequestDispatcher("/WEB-INF/views/products.jsp").forward(request, response);
    }
    
    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response) 
            throws ServletException, IOException {
        
        String action = request.getParameter("action");
        
        if ("updateStock".equals(action)) {
            handleUpdateStock(request, response);
            return;
        }
        
        // Create Product
        String name = request.getParameter("name");
        String priceStr = request.getParameter("price");
        String stockStr = request.getParameter("stock");
        
        if (name != null && priceStr != null && 
            !name.trim().isEmpty() && !priceStr.trim().isEmpty()) {
            
            try {
                BigDecimal price = new BigDecimal(priceStr);
                Integer stock = (stockStr != null && !stockStr.trim().isEmpty()) 
                    ? Integer.parseInt(stockStr) 
                    : 0;
                
                productService.createProduct(name.trim(), price, stock);
                request.setAttribute("success", "Product erfolgreich erstellt!");
            } catch (Exception e) {
                request.setAttribute("error", "Fehler beim Erstellen: " + e.getMessage());
            }
        } else {
            request.setAttribute("error", "Bitte Name und Preis ausfüllen!");
        }
        
        // Liste neu laden und anzeigen
        doGet(request, response);
    }
    
    private void handleDelete(HttpServletRequest request, HttpServletResponse response) 
            throws ServletException, IOException {
        
        String idStr = request.getParameter("id");
        if (idStr != null) {
            try {
                Long id = Long.parseLong(idStr);
                productService.deleteProduct(id);
                request.setAttribute("success", "Product gelöscht!");
            } catch (Exception e) {
                request.setAttribute("error", "Fehler beim Löschen: " + e.getMessage());
            }
        }
        
        doGet(request, response);
    }
    
    private void handleUpdateStock(HttpServletRequest request, HttpServletResponse response) 
            throws ServletException, IOException {
        
        String idStr = request.getParameter("id");
        String stockStr = request.getParameter("newStock");
        
        if (idStr != null && stockStr != null) {
            try {
                Long id = Long.parseLong(idStr);
                Integer newStock = Integer.parseInt(stockStr);
                productService.updateStock(id, newStock);
                request.setAttribute("success", "Stock aktualisiert!");
            } catch (Exception e) {
                request.setAttribute("error", "Fehler beim Aktualisieren: " + e.getMessage());
            }
        }
        
        doGet(request, response);
    }
    
    private void handleLowStock(HttpServletRequest request, HttpServletResponse response) 
            throws ServletException, IOException {
        
        List<Product> products = productService.findLowStockProducts(10);
        
        request.setAttribute("products", products);
        request.setAttribute("lowStockMode", true);
        request.getRequestDispatcher("/WEB-INF/views/products.jsp").forward(request, response);
    }
}
