package com.javafleet.servlet;

import com.javafleet.model.User;
import com.javafleet.service.UserService;
import jakarta.inject.Inject;
import jakarta.servlet.ServletException;
import jakarta.servlet.annotation.WebServlet;
import jakarta.servlet.http.HttpServlet;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.List;

/**
 * UserServlet - Web-Interface für User CRUD
 * 
 * Actions:
 * - list: Alle User anzeigen
 * - create: User erstellen
 * - delete: User löschen
 * - search: User suchen
 */
@WebServlet("/users")
public class UserServlet extends HttpServlet {
    
    @Inject
    private UserService userService;
    
    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response) 
            throws ServletException, IOException {
        
        String action = request.getParameter("action");
        
        if ("delete".equals(action)) {
            handleDelete(request, response);
            return;
        }
        
        if ("search".equals(action)) {
            handleSearch(request, response);
            return;
        }
        
        // Default: Liste anzeigen
        List<User> users = userService.findAllUsers();
        long count = userService.countUsers();
        
        request.setAttribute("users", users);
        request.setAttribute("count", count);
        request.getRequestDispatcher("/WEB-INF/views/users.jsp").forward(request, response);
    }
    
    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response) 
            throws ServletException, IOException {
        
        String username = request.getParameter("username");
        String email = request.getParameter("email");
        
        if (username != null && email != null && 
            !username.trim().isEmpty() && !email.trim().isEmpty()) {
            
            try {
                userService.createUser(username.trim(), email.trim());
                request.setAttribute("success", "User erfolgreich erstellt!");
            } catch (Exception e) {
                request.setAttribute("error", "Fehler beim Erstellen: " + e.getMessage());
            }
        } else {
            request.setAttribute("error", "Bitte alle Felder ausfüllen!");
        }
        
        // Liste neu laden und anzeigen
        doGet(request, response);
    }
    
    private void handleDelete(HttpServletRequest request, HttpServletResponse response) 
            throws ServletException, IOException {
        
        String idStr = request.getParameter("id");
        if (idStr != null) {
            try {
                Long id = Long.parseLong(idStr);
                userService.deleteUser(id);
                request.setAttribute("success", "User gelöscht!");
            } catch (Exception e) {
                request.setAttribute("error", "Fehler beim Löschen: " + e.getMessage());
            }
        }
        
        doGet(request, response);
    }
    
    private void handleSearch(HttpServletRequest request, HttpServletResponse response) 
            throws ServletException, IOException {
        
        String emailPattern = request.getParameter("email");
        if (emailPattern != null && !emailPattern.trim().isEmpty()) {
            List<User> users = userService.findUsersByEmail(emailPattern.trim());
            request.setAttribute("users", users);
            request.setAttribute("count", users.size());
            request.setAttribute("searchTerm", emailPattern);
        } else {
            // Keine Suche - alle anzeigen
            doGet(request, response);
            return;
        }
        
        request.getRequestDispatcher("/WEB-INF/views/users.jsp").forward(request, response);
    }
}
