<%@ page contentType="text/html;charset=UTF-8" language="java" %>
<%@ taglib prefix="c" uri="jakarta.tags.core" %>
<%@ taglib prefix="fmt" uri="jakarta.tags.fmt" %>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Product Management - Java Web Aufbau Tag 7</title>
    <link rel="stylesheet" href="css/style.css">
</head>
<body>
    <div class="container fade-in">
        <header>
            <h1>📦 Product Management</h1>
            <p>Challenge Lösung - Stock Verwaltung</p>
        </header>
        
        <nav>
            <a href="index.jsp">🏠 Home</a>
            <a href="users">👥 User Management</a>
            <a href="products">📦 Product Management</a>
        </nav>
        
        <!-- Success/Error Messages -->
        <c:if test="${not empty success}">
            <div class="message success">
                ✅ ${success}
            </div>
        </c:if>
        
        <c:if test="${not empty error}">
            <div class="message error">
                ❌ ${error}
            </div>
        </c:if>
        
        <!-- Create Product Form -->
        <div class="card">
            <h2>Neues Product erstellen</h2>
            <p style="margin-bottom: 20px; color: #999;">
                Challenge Requirements: Name (max 100), Price (BigDecimal), Stock (default 0)
            </p>
            
            <form method="post" action="products">
                <div class="form-group">
                    <label for="name">Product Name:</label>
                    <input type="text" id="name" name="name" 
                           placeholder="z.B. Java for Dummies" required maxlength="100">
                </div>
                
                <div class="form-group">
                    <label for="price">Preis (€):</label>
                    <input type="number" id="price" name="price" 
                           step="0.01" min="0" placeholder="z.B. 29.99" required>
                </div>
                
                <div class="form-group">
                    <label for="stock">Stock:</label>
                    <input type="number" id="stock" name="stock" 
                           min="0" placeholder="Optional (default: 0)">
                </div>
                
                <button type="submit" class="btn btn-primary">
                    ➕ Product erstellen
                </button>
            </form>
        </div>
        
        <!-- Filter Options -->
        <div class="card">
            <h2>Filter</h2>
            <div style="display: flex; gap: 15px;">
                <a href="products" class="btn btn-primary">
                    📋 Alle Products
                </a>
                <a href="products?action=lowstock" class="btn" style="background: #ffc107; color: #1a1a1a;">
                    ⚠️ Low Stock (≤10)
                </a>
            </div>
        </div>
        
        <!-- Product List -->
        <div class="card">
            <h2>Product Liste</h2>
            <p style="margin-bottom: 20px; color: #999;">
                <c:choose>
                    <c:when test="${lowStockMode}">
                        <strong>⚠️ Low Stock Products</strong> (Stock ≤ 10)
                    </c:when>
                    <c:otherwise>
                        Alle Products | Named Query: <code>Product.findAll</code>
                    </c:otherwise>
                </c:choose>
            </p>
            
            <c:choose>
                <c:when test="${empty products}">
                    <div style="background: #363636; padding: 30px; text-align: center; border-radius: 8px;">
                        <p style="font-size: 1.2em;">
                            <c:choose>
                                <c:when test="${lowStockMode}">
                                    Keine Products mit niedrigem Stock gefunden. 🎉
                                </c:when>
                                <c:otherwise>
                                    Noch keine Products vorhanden.
                                </c:otherwise>
                            </c:choose>
                        </p>
                        <p style="margin-top: 10px; color: #999;">Erstelle dein erstes Product oben! 👆</p>
                    </div>
                </c:when>
                <c:otherwise>
                    <table>
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Name</th>
                                <th>Preis</th>
                                <th>Stock</th>
                                <th>Status</th>
                                <th>Erstellt am</th>
                                <th>Aktionen</th>
                            </tr>
                        </thead>
                        <tbody>
                            <c:forEach var="product" items="${products}">
                                <tr>
                                    <td><strong>#${product.id}</strong></td>
                                    <td>${product.name}</td>
                                    <td>
                                        <strong style="color: #28a745;">
                                            <fmt:formatNumber value="${product.price}" type="currency" 
                                                            currencySymbol="€" minFractionDigits="2"/>
                                        </strong>
                                    </td>
                                    <td>
                                        <strong>${product.stock}</strong>
                                    </td>
                                    <td>
                                        <c:choose>
                                            <c:when test="${product.stock == 0}">
                                                <span class="badge badge-danger">Ausverkauft</span>
                                            </c:when>
                                            <c:when test="${product.stock <= 10}">
                                                <span class="badge badge-warning">Low Stock</span>
                                            </c:when>
                                            <c:otherwise>
                                                <span class="badge badge-success">Verfügbar</span>
                                            </c:otherwise>
                                        </c:choose>
                                    </td>
                                    <td>
                                        <c:choose>
                                            <c:when test="${not empty product.createdAt}">
                                                ${product.createdAt.toString().substring(0, 10)}
                                            </c:when>
                                            <c:otherwise>
                                                -
                                            </c:otherwise>
                                        </c:choose>
                                    </td>
                                    <td>
                                        <!-- Update Stock Form -->
                                        <form method="post" action="products" style="display: inline; background: none; padding: 0; margin-right: 5px;">
                                            <input type="hidden" name="action" value="updateStock">
                                            <input type="hidden" name="id" value="${product.id}">
                                            <input type="number" name="newStock" min="0" 
                                                   placeholder="Stock" required
                                                   style="width: 80px; padding: 5px; display: inline-block;">
                                            <button type="submit" class="btn btn-small" 
                                                    style="background: #17a2b8; color: #fff;">
                                                📝
                                            </button>
                                        </form>
                                        
                                        <!-- Delete Button -->
                                        <form method="get" action="products" style="display: inline; background: none; padding: 0;">
                                            <input type="hidden" name="action" value="delete">
                                            <input type="hidden" name="id" value="${product.id}">
                                            <button type="submit" class="btn btn-danger btn-small" 
                                                    onclick="return confirm('Product ${product.name} wirklich löschen?')">
                                                🗑️
                                            </button>
                                        </form>
                                    </td>
                                </tr>
                            </c:forEach>
                        </tbody>
                    </table>
                </c:otherwise>
            </c:choose>
        </div>
        
        <!-- Code Explanation -->
        <div class="card">
            <h2>💡 Challenge Features Demo</h2>
            
            <div style="background: #1a1a1a; padding: 20px; border-radius: 8px; margin-top: 15px; border-left: 5px solid #28a745;">
                <h3 style="color: #28a745; margin-bottom: 15px;">✅ Named Query (findAllProducts)</h3>
                <pre style="background: #000; padding: 15px; border-radius: 5px; overflow-x: auto; color: #fff; line-height: 1.6;"><code>// In Product.java Entity:
@NamedQuery(
    name = "Product.findAll",
    query = "SELECT p FROM Product p ORDER BY p.name"
)

// In ProductService:
public List&lt;Product&gt; findAllProducts() {
    return em.createNamedQuery("Product.findAll", Product.class)
        .getResultList();
}</code></pre>
            </div>
            
            <div style="background: #1a1a1a; padding: 20px; border-radius: 8px; margin-top: 15px; border-left: 5px solid #ffc107;">
                <h3 style="color: #ffc107; margin-bottom: 15px;">✅ Dirty Checking (updateStock)</h3>
                <pre style="background: #000; padding: 15px; border-radius: 5px; overflow-x: auto; color: #fff; line-height: 1.6;"><code>public void updateStock(Long id, Integer newStock) {
    Product product = em.find(Product.class, id);  // MANAGED
    if (product != null) {
        product.setStock(newStock);
        // KEIN merge() nötig!
        // JPA tracked die Änderung automatisch
        // UPDATE beim Transaction-Commit!
    }
}</code></pre>
            </div>
            
            <div style="background: #1a1a1a; padding: 20px; border-radius: 8px; margin-top: 15px; border-left: 5px solid #17a2b8;">
                <h3 style="color: #17a2b8; margin-bottom: 15px;">✅ @PrePersist Callback</h3>
                <pre style="background: #000; padding: 15px; border-radius: 5px; overflow-x: auto; color: #fff; line-height: 1.6;"><code>// In Product.java:
@PrePersist
protected void onCreate() {
    createdAt = LocalDateTime.now();  // Automatisch!
    if (stock == null) {
        stock = 0;  // Default 0
    }
}</code></pre>
            </div>
            
            <div style="background: #1a1a1a; padding: 20px; border-radius: 8px; margin-top: 15px; border-left: 5px solid #dc3545;">
                <h3 style="color: #dc3545; margin-bottom: 15px;">✅ Custom JPQL Query (Low Stock)</h3>
                <pre style="background: #000; padding: 15px; border-radius: 5px; overflow-x: auto; color: #fff; line-height: 1.6;"><code>public List&lt;Product&gt; findLowStockProducts(Integer threshold) {
    return em.createQuery(
        "SELECT p FROM Product p WHERE p.stock <= :threshold " +
        "ORDER BY p.stock", Product.class)
        .setParameter("threshold", threshold)
        .getResultList();
}</code></pre>
            </div>
        </div>
        
        <footer>
            <p><strong>Product Management Demo</strong></p>
            <p style="margin-top: 10px;">
                Challenge Lösung | Java Web Aufbau - Tag 7
            </p>
        </footer>
    </div>
</body>
</html>
