package de.javafleet.oop.model;

/**
 * Gekapselte Auto-Klasse für Tag 3: Datenkapselung & Sichtbarkeit.
 * 
 * Demonstriert:
 * - Private Attribute
 * - Getter und Setter
 * - Validierung in Settern
 * - JavaBeans-Konvention
 * - Kontrollierte Zustandsänderungen
 * 
 * @author Nova Trent
 * @version 3.0
 */
public class Auto {
    
    // ========================================
    // PRIVATE ATTRIBUTE
    // ========================================
    
    private String farbe;
    private String marke;
    private int ps;
    private int kilometerstand;
    private double tankinhalt;
    private final double tankkapazitaet;  // Kann nicht geändert werden!
    private boolean motorLaeuft;
    
    // ========================================
    // KONSTRUKTOR (wird in Tag 4 erklärt)
    // ========================================
    
    /**
     * Erstellt ein neues Auto mit Standardwerten.
     */
    public Auto() {
        this.kilometerstand = 0;
        this.tankinhalt = 0;
        this.tankkapazitaet = 50.0;  // 50 Liter Standard
        this.motorLaeuft = false;
    }
    
    // ========================================
    // GETTER (Werte lesen)
    // ========================================
    
    public String getFarbe() {
        return farbe;
    }
    
    public String getMarke() {
        return marke;
    }
    
    public int getPs() {
        return ps;
    }
    
    public int getKilometerstand() {
        return kilometerstand;
    }
    
    public double getTankinhalt() {
        return tankinhalt;
    }
    
    public double getTankkapazitaet() {
        return tankkapazitaet;
    }
    
    /**
     * Getter für boolean: isXxx() statt getXxx()
     */
    public boolean isMotorLaeuft() {
        return motorLaeuft;
    }
    
    // ========================================
    // SETTER MIT VALIDIERUNG
    // ========================================
    
    /**
     * Setzt die Farbe des Autos.
     * 
     * @param farbe Die neue Farbe (darf nicht null oder leer sein)
     * @throws IllegalArgumentException wenn Farbe ungültig
     */
    public void setFarbe(String farbe) {
        if (farbe == null || farbe.isBlank()) {
            throw new IllegalArgumentException("Farbe darf nicht leer sein!");
        }
        this.farbe = farbe;
    }
    
    /**
     * Setzt die Marke des Autos.
     * 
     * @param marke Die neue Marke (darf nicht null oder leer sein)
     * @throws IllegalArgumentException wenn Marke ungültig
     */
    public void setMarke(String marke) {
        if (marke == null || marke.isBlank()) {
            throw new IllegalArgumentException("Marke darf nicht leer sein!");
        }
        this.marke = marke;
    }
    
    /**
     * Setzt die PS-Zahl des Autos.
     * 
     * @param ps Die PS-Zahl (muss zwischen 1 und 2000 liegen)
     * @throws IllegalArgumentException wenn PS ungültig
     */
    public void setPs(int ps) {
        if (ps < 1) {
            throw new IllegalArgumentException("PS muss mindestens 1 sein!");
        }
        if (ps > 2000) {
            throw new IllegalArgumentException("PS darf maximal 2000 sein!");
        }
        this.ps = ps;
    }
    
    // ========================================
    // KEIN SETTER FÜR KILOMETERSTAND!
    // Stattdessen: Kontrollierte Methode
    // ========================================
    
    /**
     * Fährt eine bestimmte Strecke.
     * Kilometerstand kann nur ERHÖHT werden, nicht direkt gesetzt!
     * 
     * @param km Die zu fahrenden Kilometer (muss positiv sein)
     * @throws IllegalArgumentException wenn km negativ
     * @throws IllegalStateException wenn Motor nicht läuft
     */
    public void fahren(int km) {
        if (km < 0) {
            throw new IllegalArgumentException("Kilometer dürfen nicht negativ sein!");
        }
        if (!motorLaeuft) {
            throw new IllegalStateException("Motor muss erst gestartet werden!");
        }
        
        double verbrauch = km * 0.07;  // 7 Liter / 100km
        if (verbrauch > tankinhalt) {
            throw new IllegalStateException(
                "Nicht genug Benzin! Brauche " + verbrauch + "L, habe " + tankinhalt + "L"
            );
        }
        
        this.kilometerstand += km;
        this.tankinhalt -= verbrauch;
        
        System.out.println("✓ Gefahren: " + km + " km | Kilometerstand: " + 
            kilometerstand + " km | Tank: " + String.format("%.1f", tankinhalt) + " L");
    }
    
    // ========================================
    // TANKINHALT: Kontrollierte Methoden
    // ========================================
    
    /**
     * Tankt das Auto auf.
     * 
     * @param liter Die zu tankenden Liter (muss positiv sein)
     * @throws IllegalArgumentException wenn liter negativ
     */
    public void tanken(double liter) {
        if (liter <= 0) {
            throw new IllegalArgumentException("Tankmenge muss positiv sein!");
        }
        
        double neuerStand = tankinhalt + liter;
        if (neuerStand > tankkapazitaet) {
            double tatsaechlich = tankkapazitaet - tankinhalt;
            this.tankinhalt = tankkapazitaet;
            System.out.println("⚠️ Tank voll! Nur " + String.format("%.1f", tatsaechlich) + 
                " Liter getankt. Tank: " + tankkapazitaet + " L");
        } else {
            this.tankinhalt = neuerStand;
            System.out.println("✓ Getankt: " + liter + " L | Tank: " + 
                String.format("%.1f", tankinhalt) + " L");
        }
    }
    
    // ========================================
    // MOTOR: Zustandsänderungen mit Logik
    // ========================================
    
    /**
     * Startet den Motor.
     */
    public void starteMotor() {
        if (motorLaeuft) {
            System.out.println("⚠️ Motor läuft bereits!");
            return;
        }
        if (tankinhalt <= 0) {
            throw new IllegalStateException("Kein Benzin im Tank!");
        }
        
        this.motorLaeuft = true;
        System.out.println("🚗 Motor gestartet!");
    }
    
    /**
     * Stoppt den Motor.
     */
    public void stoppeMotor() {
        if (!motorLaeuft) {
            System.out.println("⚠️ Motor ist bereits aus!");
            return;
        }
        
        this.motorLaeuft = false;
        System.out.println("🔇 Motor gestoppt.");
    }
    
    // ========================================
    // INFO-METHODE
    // ========================================
    
    /**
     * Gibt alle Informationen über das Auto aus.
     */
    public void info() {
        System.out.println("╔════════════════════════════════════╗");
        System.out.println("║         AUTO-INFORMATION           ║");
        System.out.println("╠════════════════════════════════════╣");
        System.out.println("║ Marke:         " + pad(marke) + "║");
        System.out.println("║ Farbe:         " + pad(farbe) + "║");
        System.out.println("║ Leistung:      " + pad(ps + " PS") + "║");
        System.out.println("║ Kilometerstand:" + pad(kilometerstand + " km") + "║");
        System.out.println("║ Tank:          " + pad(String.format("%.1f / %.0f L", tankinhalt, tankkapazitaet)) + "║");
        System.out.println("║ Motor:         " + pad(motorLaeuft ? "läuft" : "aus") + "║");
        System.out.println("╚════════════════════════════════════╝");
    }
    
    private String pad(String s) {
        if (s == null) s = "-";
        return String.format("%-19s", s);
    }
}
