package de.javafleet.oop.model;

/**
 * Auto-Klasse für Tag 4: Konstruktoren.
 * 
 * Demonstriert:
 * - Konstruktor mit Validierung (Hauptkonstruktor)
 * - Konstruktorüberladung
 * - Konstruktorverkettung mit this()
 * - final-Attribute im Konstruktor
 * 
 * @author Nova Trent
 * @version 4.0
 */
public class Auto {
    
    // ========================================
    // ATTRIBUTE
    // ========================================
    
    private final String fahrgestellnummer;  // Unveränderlich!
    private String farbe;
    private int ps;
    private String marke;
    private int kilometerstand;
    
    // ========================================
    // KONSTRUKTOREN (mit Verkettung)
    // ========================================
    
    /**
     * HAUPTKONSTRUKTOR — Hier passiert die echte Arbeit!
     * Alle anderen Konstruktoren rufen diesen auf.
     * 
     * @param fahrgestellnummer Eindeutige ID (17 Zeichen)
     * @param farbe Die Farbe des Autos
     * @param ps Die Leistung in PS
     * @param marke Die Marke
     * @throws IllegalArgumentException bei ungültigen Werten
     */
    public Auto(String fahrgestellnummer, String farbe, int ps, String marke) {
        // === VALIDIERUNG ===
        
        // Fahrgestellnummer prüfen
        if (fahrgestellnummer == null || fahrgestellnummer.isBlank()) {
            throw new IllegalArgumentException("Fahrgestellnummer darf nicht leer sein!");
        }
        if (fahrgestellnummer.length() != 17) {
            throw new IllegalArgumentException(
                "Fahrgestellnummer muss genau 17 Zeichen haben! (Hatte: " + 
                fahrgestellnummer.length() + ")"
            );
        }
        
        // Farbe prüfen
        if (farbe == null || farbe.isBlank()) {
            throw new IllegalArgumentException("Farbe darf nicht leer sein!");
        }
        
        // PS prüfen
        if (ps < 1) {
            throw new IllegalArgumentException("PS muss mindestens 1 sein!");
        }
        if (ps > 2000) {
            throw new IllegalArgumentException("PS darf maximal 2000 sein!");
        }
        
        // Marke prüfen
        if (marke == null || marke.isBlank()) {
            throw new IllegalArgumentException("Marke darf nicht leer sein!");
        }
        
        // === INITIALISIERUNG ===
        this.fahrgestellnummer = fahrgestellnummer;
        this.farbe = farbe;
        this.ps = ps;
        this.marke = marke;
        this.kilometerstand = 0;
        
        System.out.println("✓ Auto erstellt: " + marke + " " + farbe + " (" + ps + " PS)");
    }
    
    /**
     * Konstruktor ohne Marke — setzt "Unbekannt" als Default.
     * 
     * @param fahrgestellnummer Eindeutige ID
     * @param farbe Die Farbe
     * @param ps Die Leistung in PS
     */
    public Auto(String fahrgestellnummer, String farbe, int ps) {
        this(fahrgestellnummer, farbe, ps, "Unbekannt");
        //   ↑ Ruft den Hauptkonstruktor auf!
    }
    
    /**
     * Konstruktor nur mit Fahrgestellnummer — setzt alle Defaults.
     * 
     * @param fahrgestellnummer Eindeutige ID
     */
    public Auto(String fahrgestellnummer) {
        this(fahrgestellnummer, "Weiß", 100);
        //   ↑ Ruft den 3-Parameter-Konstruktor auf!
    }
    
    /**
     * COPY-KONSTRUKTOR — Erstellt eine Kopie eines Autos.
     * Die Fahrgestellnummer wird mit "-COPY" erweitert.
     * 
     * @param original Das zu kopierende Auto
     */
    public Auto(Auto original) {
        this(
            original.fahrgestellnummer + "-COPY".substring(0, 
                Math.min(5, 17 - original.fahrgestellnummer.length())),
            original.farbe,
            original.ps,
            original.marke
        );
        this.kilometerstand = original.kilometerstand;
        System.out.println("  (Kopie von Fahrgestell: " + original.fahrgestellnummer + ")");
    }
    
    // ========================================
    // GETTER
    // ========================================
    
    public String getFahrgestellnummer() {
        return fahrgestellnummer;
    }
    
    public String getFarbe() {
        return farbe;
    }
    
    public int getPs() {
        return ps;
    }
    
    public String getMarke() {
        return marke;
    }
    
    public int getKilometerstand() {
        return kilometerstand;
    }
    
    // ========================================
    // SETTER (nur für nicht-finale Attribute)
    // ========================================
    
    public void setFarbe(String farbe) {
        if (farbe == null || farbe.isBlank()) {
            throw new IllegalArgumentException("Farbe darf nicht leer sein!");
        }
        this.farbe = farbe;
    }
    
    // Kein Setter für fahrgestellnummer (final!)
    
    // ========================================
    // METHODEN
    // ========================================
    
    /**
     * Fährt eine bestimmte Strecke.
     * 
     * @param km Die zu fahrenden Kilometer
     */
    public void fahren(int km) {
        if (km < 0) {
            throw new IllegalArgumentException("Kilometer dürfen nicht negativ sein!");
        }
        this.kilometerstand += km;
        System.out.println("Gefahren: " + km + " km | Gesamt: " + kilometerstand + " km");
    }
    
    /**
     * Gibt alle Informationen über das Auto aus.
     */
    public void info() {
        System.out.println("╔════════════════════════════════════════╗");
        System.out.println("║           AUTO-INFORMATION             ║");
        System.out.println("╠════════════════════════════════════════╣");
        System.out.println("║ Fahrgestell:    " + pad(fahrgestellnummer, 22) + "║");
        System.out.println("║ Marke:          " + pad(marke, 22) + "║");
        System.out.println("║ Farbe:          " + pad(farbe, 22) + "║");
        System.out.println("║ Leistung:       " + pad(ps + " PS", 22) + "║");
        System.out.println("║ Kilometerstand: " + pad(kilometerstand + " km", 22) + "║");
        System.out.println("╚════════════════════════════════════════╝");
    }
    
    private String pad(String s, int len) {
        return String.format("%-" + len + "s", s);
    }
}
