package de.javafleet.oop.model;

/**
 * Bankkonto-Klasse — Lösung der Tag 3 Challenge!
 * 
 * Demonstriert:
 * - Kapselung mit privaten Attributen
 * - Validierung im Konstruktor und in Methoden
 * - final-Attribute für unveränderliche Werte
 * - Sicherheitsprüfungen (PIN)
 * 
 * @author Nova Trent
 * @version 1.0
 */
public class Bankkonto {
    
    // ========================================
    // PRIVATE ATTRIBUTE
    // ========================================
    
    private final String kontonummer;  // Unveränderlich nach Erstellung!
    private double kontostand;
    private String pin;                 // Kein Getter! Nur intern verwendbar.
    
    // ========================================
    // KONSTRUKTOREN
    // ========================================
    
    /**
     * Hauptkonstruktor mit allen Parametern.
     * 
     * @param kontonummer Die Kontonummer (IBAN-Format oder beliebig)
     * @param pin Die PIN (muss genau 4 Ziffern haben)
     * @param startguthaben Das Startguthaben (muss >= 0 sein)
     * @throws IllegalArgumentException bei ungültigen Werten
     */
    public Bankkonto(String kontonummer, String pin, double startguthaben) {
        // Kontonummer validieren
        if (kontonummer == null || kontonummer.isBlank()) {
            throw new IllegalArgumentException("Kontonummer darf nicht leer sein!");
        }
        
        // PIN validieren (genau 4 Ziffern)
        if (pin == null || !pin.matches("\\d{4}")) {
            throw new IllegalArgumentException("PIN muss genau 4 Ziffern haben!");
        }
        
        // Startguthaben validieren
        if (startguthaben < 0) {
            throw new IllegalArgumentException("Startguthaben darf nicht negativ sein!");
        }
        
        this.kontonummer = kontonummer;
        this.pin = pin;
        this.kontostand = startguthaben;
        
        System.out.println("✓ Konto eröffnet: " + kontonummer);
        System.out.println("  Startguthaben: " + startguthaben + " €");
    }
    
    /**
     * Konstruktor ohne Startguthaben (beginnt bei 0 €).
     * 
     * @param kontonummer Die Kontonummer
     * @param pin Die PIN (4 Ziffern)
     */
    public Bankkonto(String kontonummer, String pin) {
        this(kontonummer, pin, 0.0);
    }
    
    // ========================================
    // GETTER (kein Getter für PIN!)
    // ========================================
    
    public String getKontonummer() {
        return kontonummer;
    }
    
    public double getKontostand() {
        return kontostand;
    }
    
    // ⚠️ Absichtlich KEIN getPin()! Sicherheit!
    
    // ========================================
    // GESCHÄFTSMETHODEN
    // ========================================
    
    /**
     * Zahlt einen Betrag auf das Konto ein.
     * 
     * @param betrag Der einzuzahlende Betrag (muss positiv sein)
     * @throws IllegalArgumentException wenn Betrag <= 0
     */
    public void einzahlen(double betrag) {
        if (betrag <= 0) {
            throw new IllegalArgumentException("Betrag muss positiv sein!");
        }
        
        kontostand += betrag;
        System.out.println("✓ Eingezahlt: " + String.format("%.2f", betrag) + " €");
        System.out.println("  Neuer Kontostand: " + String.format("%.2f", kontostand) + " €");
    }
    
    /**
     * Hebt einen Betrag vom Konto ab.
     * 
     * @param betrag Der abzuhebende Betrag
     * @param pinEingabe Die eingegebene PIN zur Verifizierung
     * @throws SecurityException wenn PIN falsch
     * @throws IllegalArgumentException wenn Betrag ungültig oder nicht genug Guthaben
     */
    public void abheben(double betrag, String pinEingabe) {
        // PIN prüfen
        if (!pin.equals(pinEingabe)) {
            System.out.println("❌ Transaktion abgelehnt!");
            throw new SecurityException("Falsche PIN!");
        }
        
        // Betrag validieren
        if (betrag <= 0) {
            throw new IllegalArgumentException("Betrag muss positiv sein!");
        }
        
        // Deckung prüfen
        if (betrag > kontostand) {
            throw new IllegalArgumentException(
                "Nicht genug Guthaben! Verfügbar: " + String.format("%.2f", kontostand) + " €"
            );
        }
        
        kontostand -= betrag;
        System.out.println("✓ Abgehoben: " + String.format("%.2f", betrag) + " €");
        System.out.println("  Neuer Kontostand: " + String.format("%.2f", kontostand) + " €");
    }
    
    /**
     * Ändert die PIN des Kontos.
     * 
     * @param altePin Die aktuelle PIN zur Verifizierung
     * @param neuePin Die neue PIN (muss 4 Ziffern haben)
     * @throws SecurityException wenn alte PIN falsch
     * @throws IllegalArgumentException wenn neue PIN ungültig
     */
    public void pinAendern(String altePin, String neuePin) {
        // Alte PIN prüfen
        if (!pin.equals(altePin)) {
            System.out.println("❌ PIN-Änderung abgelehnt!");
            throw new SecurityException("Alte PIN ist falsch!");
        }
        
        // Neue PIN validieren
        if (neuePin == null || !neuePin.matches("\\d{4}")) {
            throw new IllegalArgumentException("Neue PIN muss genau 4 Ziffern haben!");
        }
        
        // Gleiche PIN verhindern
        if (neuePin.equals(altePin)) {
            throw new IllegalArgumentException("Neue PIN muss sich von alter PIN unterscheiden!");
        }
        
        this.pin = neuePin;
        System.out.println("✓ PIN wurde erfolgreich geändert.");
    }
    
    /**
     * Gibt Kontoinformationen aus (ohne PIN!).
     */
    public void info() {
        System.out.println("┌─────────────────────────────────┐");
        System.out.println("│       KONTOINFORMATION          │");
        System.out.println("├─────────────────────────────────┤");
        System.out.println("│ Kontonummer: " + pad(kontonummer, 18) + "│");
        System.out.println("│ Kontostand:  " + pad(String.format("%.2f €", kontostand), 18) + "│");
        System.out.println("│ PIN:         " + pad("****", 18) + "│");
        System.out.println("└─────────────────────────────────┘");
    }
    
    private String pad(String s, int len) {
        return String.format("%-" + len + "s", s);
    }
}
