package de.javafleet.oop.model;

/**
 * Auto-Klasse für Tag 5: Static & Konstanten.
 * 
 * Demonstriert:
 * - Static Klassenvariable (anzahlAutos)
 * - Static Konstanten (MAX_PS, MIN_PS)
 * - Static Methoden (getAnzahlAutos)
 * - Kombination von static und Instanzvariablen
 * 
 * @author Nova Trent
 * @version 5.0
 */
public class Auto {
    
    // ========================================
    // STATIC KONSTANTEN (static final)
    // ========================================
    
    /** Maximale PS-Zahl */
    public static final int MAX_PS = 2000;
    
    /** Minimale PS-Zahl */
    public static final int MIN_PS = 1;
    
    /** Standard-Farbe für neue Autos */
    public static final String DEFAULT_FARBE = "Weiß";
    
    /** Standard-Marke für neue Autos */
    public static final String DEFAULT_MARKE = "Unbekannt";
    
    // ========================================
    // STATIC KLASSENVARIABLE
    // ========================================
    
    /** Zählt die Anzahl aller erstellten Autos */
    private static int anzahlAutos = 0;
    
    // ========================================
    // INSTANZVARIABLEN
    // ========================================
    
    private final int autoNummer;  // Eindeutige Nummer dieses Autos
    private String farbe;
    private int ps;
    private String marke;
    private int kilometerstand;
    
    // ========================================
    // KONSTRUKTOREN
    // ========================================
    
    /**
     * Hauptkonstruktor mit allen Parametern.
     */
    public Auto(String farbe, int ps, String marke) {
        // Validierung
        if (farbe == null || farbe.isBlank()) {
            throw new IllegalArgumentException("Farbe darf nicht leer sein!");
        }
        if (ps < MIN_PS || ps > MAX_PS) {
            throw new IllegalArgumentException(
                "PS muss zwischen " + MIN_PS + " und " + MAX_PS + " liegen!"
            );
        }
        if (marke == null || marke.isBlank()) {
            throw new IllegalArgumentException("Marke darf nicht leer sein!");
        }
        
        // Static Zähler erhöhen
        anzahlAutos++;
        
        // Instanzvariablen setzen
        this.autoNummer = anzahlAutos;  // Eindeutige Nummer
        this.farbe = farbe;
        this.ps = ps;
        this.marke = marke;
        this.kilometerstand = 0;
        
        System.out.println("✓ Auto #" + autoNummer + " erstellt: " + 
            marke + " " + farbe + " (" + ps + " PS)");
    }
    
    /**
     * Konstruktor mit Default-Marke.
     */
    public Auto(String farbe, int ps) {
        this(farbe, ps, DEFAULT_MARKE);
    }
    
    /**
     * Konstruktor mit allen Defaults.
     */
    public Auto() {
        this(DEFAULT_FARBE, 100, DEFAULT_MARKE);
    }
    
    // ========================================
    // STATIC METHODEN
    // ========================================
    
    /**
     * Gibt die Anzahl aller erstellten Autos zurück.
     * STATIC: Kann ohne Objekt aufgerufen werden!
     * 
     * @return Anzahl der Autos
     */
    public static int getAnzahlAutos() {
        return anzahlAutos;
    }
    
    /**
     * Setzt den Zähler zurück (z.B. für Tests).
     */
    public static void resetZaehler() {
        anzahlAutos = 0;
        System.out.println("⚠️ Auto-Zähler wurde zurückgesetzt.");
    }
    
    /**
     * Prüft, ob eine PS-Zahl gültig ist.
     * STATIC: Braucht kein Objekt, reine Berechnung.
     */
    public static boolean istGueltigePS(int ps) {
        return ps >= MIN_PS && ps <= MAX_PS;
    }
    
    // ========================================
    // INSTANZMETHODEN (Getter)
    // ========================================
    
    public int getAutoNummer() {
        return autoNummer;
    }
    
    public String getFarbe() {
        return farbe;
    }
    
    public int getPs() {
        return ps;
    }
    
    public String getMarke() {
        return marke;
    }
    
    public int getKilometerstand() {
        return kilometerstand;
    }
    
    // ========================================
    // INSTANZMETHODEN (Aktionen)
    // ========================================
    
    public void fahren(int km) {
        if (km < 0) {
            throw new IllegalArgumentException("Kilometer dürfen nicht negativ sein!");
        }
        this.kilometerstand += km;
    }
    
    public void info() {
        System.out.println("╔════════════════════════════════════════╗");
        System.out.println("║           AUTO #" + String.format("%-3d", autoNummer) + "                    ║");
        System.out.println("╠════════════════════════════════════════╣");
        System.out.println("║ Marke:          " + pad(marke, 22) + "║");
        System.out.println("║ Farbe:          " + pad(farbe, 22) + "║");
        System.out.println("║ Leistung:       " + pad(ps + " PS", 22) + "║");
        System.out.println("║ Kilometerstand: " + pad(kilometerstand + " km", 22) + "║");
        System.out.println("╚════════════════════════════════════════╝");
    }
    
    private String pad(String s, int len) {
        return String.format("%-" + len + "s", s);
    }
}
