package de.javafleet.oop.model;

/**
 * Mitarbeiter-Klasse — Lösung der Tag 4 Challenge!
 * 
 * Demonstriert:
 * - Konstruktorverkettung
 * - Validierung (Personalnummer-Format)
 * - final-Attribute
 * 
 * @author Nova Trent
 * @version 1.0
 */
public class Mitarbeiter {
    
    // ========================================
    // STATIC KONSTANTEN
    // ========================================
    
    public static final double MINDEST_GEHALT = 1500.0;
    public static final double DEFAULT_GEHALT = 3000.0;
    public static final String DEFAULT_ABTEILUNG = "Allgemein";
    
    // ========================================
    // INSTANZVARIABLEN
    // ========================================
    
    private final String personalnummer;  // Unveränderlich!
    private String name;
    private String abteilung;
    private double gehalt;
    
    // ========================================
    // KONSTRUKTOREN (mit Verkettung)
    // ========================================
    
    /**
     * HAUPTKONSTRUKTOR mit allen Parametern.
     * 
     * @param personalnummer Format: P + 4 Ziffern (z.B. "P1234")
     * @param name Der Name des Mitarbeiters
     * @param abteilung Die Abteilung
     * @param gehalt Das Gehalt (mindestens MINDEST_GEHALT)
     */
    public Mitarbeiter(String personalnummer, String name, String abteilung, double gehalt) {
        // Personalnummer validieren: P + 4 Ziffern
        if (personalnummer == null || !personalnummer.matches("P\\d{4}")) {
            throw new IllegalArgumentException(
                "Personalnummer muss Format 'P1234' haben! (Erhalten: " + personalnummer + ")"
            );
        }
        
        // Name validieren
        if (name == null || name.isBlank()) {
            throw new IllegalArgumentException("Name darf nicht leer sein!");
        }
        
        // Abteilung validieren
        if (abteilung == null || abteilung.isBlank()) {
            throw new IllegalArgumentException("Abteilung darf nicht leer sein!");
        }
        
        // Gehalt validieren
        if (gehalt < MINDEST_GEHALT) {
            throw new IllegalArgumentException(
                "Gehalt muss mindestens " + MINDEST_GEHALT + "€ sein!"
            );
        }
        
        this.personalnummer = personalnummer;
        this.name = name;
        this.abteilung = abteilung;
        this.gehalt = gehalt;
        
        System.out.println("✓ Mitarbeiter erstellt: " + personalnummer + " - " + name);
    }
    
    /**
     * Konstruktor ohne Gehalt (Default: DEFAULT_GEHALT).
     */
    public Mitarbeiter(String personalnummer, String name, String abteilung) {
        this(personalnummer, name, abteilung, DEFAULT_GEHALT);
    }
    
    /**
     * Konstruktor nur mit Personalnummer und Name.
     */
    public Mitarbeiter(String personalnummer, String name) {
        this(personalnummer, name, DEFAULT_ABTEILUNG);
    }
    
    // ========================================
    // GETTER
    // ========================================
    
    public String getPersonalnummer() {
        return personalnummer;
    }
    
    public String getName() {
        return name;
    }
    
    public String getAbteilung() {
        return abteilung;
    }
    
    public double getGehalt() {
        return gehalt;
    }
    
    // ========================================
    // SETTER (nur für änderbare Attribute)
    // ========================================
    
    public void setName(String name) {
        if (name == null || name.isBlank()) {
            throw new IllegalArgumentException("Name darf nicht leer sein!");
        }
        this.name = name;
    }
    
    public void setAbteilung(String abteilung) {
        if (abteilung == null || abteilung.isBlank()) {
            throw new IllegalArgumentException("Abteilung darf nicht leer sein!");
        }
        this.abteilung = abteilung;
    }
    
    public void setGehalt(double gehalt) {
        if (gehalt < MINDEST_GEHALT) {
            throw new IllegalArgumentException(
                "Gehalt muss mindestens " + MINDEST_GEHALT + "€ sein!"
            );
        }
        this.gehalt = gehalt;
    }
    
    // ========================================
    // INFO
    // ========================================
    
    public void info() {
        System.out.println("┌─────────────────────────────────┐");
        System.out.println("│         MITARBEITER             │");
        System.out.println("├─────────────────────────────────┤");
        System.out.println("│ PNR:      " + pad(personalnummer, 20) + "│");
        System.out.println("│ Name:     " + pad(name, 20) + "│");
        System.out.println("│ Abteilung:" + pad(abteilung, 20) + "│");
        System.out.println("│ Gehalt:   " + pad(String.format("%.2f €", gehalt), 20) + "│");
        System.out.println("└─────────────────────────────────┘");
    }
    
    private String pad(String s, int len) {
        return String.format("%-" + len + "s", s);
    }
}
