package de.javafleet.oop.model;

/**
 * Konto-Klasse — Lösung der Tag 5 Challenge!
 * 
 * Demonstriert:
 * - Static Konstanten (MINDEST_EINLAGE, MAX_TAGESABHEBUNG, IBAN_PREFIX)
 * - Static Zähler (anzahlKonten)
 * - Static Methoden (getAnzahlKonten, generiereIBAN)
 * 
 * @author Nova Trent
 * @version 1.0
 */
public class Konto {
    
    // ========================================
    // STATIC KONSTANTEN
    // ========================================
    
    /** Mindesteinlage bei Kontoeröffnung */
    public static final double MINDEST_EINLAGE = 100.0;
    
    /** Maximale Tagesabhebung */
    public static final double MAX_TAGESABHEBUNG = 1000.0;
    
    /** IBAN-Präfix für Deutschland */
    public static final String IBAN_PREFIX = "DE";
    
    // ========================================
    // STATIC KLASSENVARIABLE
    // ========================================
    
    /** Zählt alle erstellten Konten */
    private static int anzahlKonten = 0;
    
    // ========================================
    // INSTANZVARIABLEN
    // ========================================
    
    private final String iban;
    private double kontostand;
    private double heuteAbgehoben;
    
    // ========================================
    // KONSTRUKTOR
    // ========================================
    
    /**
     * Erstellt ein neues Konto mit Startguthaben.
     * 
     * @param startguthaben Muss mindestens MINDEST_EINLAGE sein
     */
    public Konto(double startguthaben) {
        if (startguthaben < MINDEST_EINLAGE) {
            throw new IllegalArgumentException(
                "Mindesteinlage ist " + MINDEST_EINLAGE + "€!"
            );
        }
        
        this.iban = generiereIBAN();
        this.kontostand = startguthaben;
        this.heuteAbgehoben = 0;
        
        System.out.println("✓ Konto eröffnet: " + iban);
        System.out.println("  Startguthaben: " + startguthaben + "€");
    }
    
    // ========================================
    // STATIC METHODEN
    // ========================================
    
    /**
     * Gibt die Anzahl aller erstellten Konten zurück.
     */
    public static int getAnzahlKonten() {
        return anzahlKonten;
    }
    
    /**
     * Generiert eine eindeutige IBAN.
     * Format: DE + 20-stellige Nummer basierend auf anzahlKonten
     */
    private static String generiereIBAN() {
        anzahlKonten++;
        return IBAN_PREFIX + String.format("%020d", anzahlKonten);
    }
    
    /**
     * Setzt den Konten-Zähler zurück (für Tests).
     */
    public static void resetZaehler() {
        anzahlKonten = 0;
        System.out.println("⚠️ Konten-Zähler wurde zurückgesetzt.");
    }
    
    // ========================================
    // GETTER
    // ========================================
    
    public String getIban() {
        return iban;
    }
    
    public double getKontostand() {
        return kontostand;
    }
    
    // ========================================
    // GESCHÄFTSMETHODEN
    // ========================================
    
    /**
     * Zahlt einen Betrag ein.
     */
    public void einzahlen(double betrag) {
        if (betrag <= 0) {
            throw new IllegalArgumentException("Betrag muss positiv sein!");
        }
        kontostand += betrag;
        System.out.println("✓ Eingezahlt: " + betrag + "€ | Kontostand: " + kontostand + "€");
    }
    
    /**
     * Hebt einen Betrag ab (mit Tageslimit-Prüfung).
     */
    public void abheben(double betrag) {
        if (betrag <= 0) {
            throw new IllegalArgumentException("Betrag muss positiv sein!");
        }
        if (betrag > kontostand) {
            throw new IllegalArgumentException("Nicht genug Guthaben!");
        }
        if (heuteAbgehoben + betrag > MAX_TAGESABHEBUNG) {
            throw new IllegalArgumentException(
                "Tageslimit überschritten! Bereits abgehoben: " + heuteAbgehoben + 
                "€, Limit: " + MAX_TAGESABHEBUNG + "€"
            );
        }
        
        kontostand -= betrag;
        heuteAbgehoben += betrag;
        System.out.println("✓ Abgehoben: " + betrag + "€ | Kontostand: " + kontostand + "€");
    }
    
    /**
     * Setzt das Tageslimit zurück (neuer Tag).
     */
    public void neuerTag() {
        heuteAbgehoben = 0;
        System.out.println("📅 Neuer Tag — Tageslimit zurückgesetzt.");
    }
    
    /**
     * Gibt Kontoinformationen aus.
     */
    public void info() {
        System.out.println("┌─────────────────────────────────────┐");
        System.out.println("│           KONTOINFORMATION          │");
        System.out.println("├─────────────────────────────────────┤");
        System.out.println("│ IBAN:        " + iban + " │");
        System.out.println("│ Kontostand:  " + String.format("%10.2f €", kontostand) + "           │");
        System.out.println("│ Heute abgeh: " + String.format("%10.2f €", heuteAbgehoben) + "           │");
        System.out.println("└─────────────────────────────────────┘");
    }
}
