package de.javafleet.oop;

import de.javafleet.oop.model.*;
import de.javafleet.oop.enums.*;
import de.javafleet.oop.exceptions.*;

/**
 * Einstiegspunkt für Tag 10: Ausnahmebehandlung.
 * 
 * Demonstriert:
 * - try-catch-finally
 * - Checked vs. Unchecked Exceptions
 * - Eigene Exceptions
 * - Best Practices
 * 
 * @author Nova Trent
 */
public class Main {
    
    public static void main(String[] args) {
        System.out.println("╔═══════════════════════════════════════════════════╗");
        System.out.println("║     Java OOP - Tag 10: Ausnahmebehandlung 🔴      ║");
        System.out.println("║               --- FINALE ---                      ║");
        System.out.println("╚═══════════════════════════════════════════════════╝");
        System.out.println();
        
        // ========================================
        // TEIL 1: Zahlungsmittel (Lösung Tag 9)
        // ========================================
        
        System.out.println("═══ TEIL 1: Zahlungsmittel (Lösung Tag 9) ═══");
        System.out.println();
        
        Kreditkarte karte = new Kreditkarte("1234567890123456", 2000);
        PayPal paypal = new PayPal("nova@javafleet.de", 500);
        
        System.out.println();
        karte.bezahlen(150);
        paypal.bezahlen(75);
        
        System.out.println();
        System.out.println("Währungen:");
        for (Waehrung w : Waehrung.values()) {
            System.out.println("  " + w + ": " + w.formatieren(100));
        }
        
        System.out.println();
        
        // ========================================
        // TEIL 2: Einfaches try-catch
        // ========================================
        
        System.out.println("═══ TEIL 2: Einfaches try-catch ═══");
        System.out.println();
        
        try {
            int[] zahlen = {1, 2, 3};
            System.out.println("Zugriff auf Index 99...");
            System.out.println(zahlen[99]);  // 💥
        } catch (ArrayIndexOutOfBoundsException e) {
            System.out.println("❌ Gefangen: " + e.getClass().getSimpleName());
            System.out.println("   Message: " + e.getMessage());
        }
        System.out.println("✅ Programm läuft weiter!");
        
        System.out.println();
        
        // ========================================
        // TEIL 3: try-catch-finally
        // ========================================
        
        System.out.println("═══ TEIL 3: try-catch-finally ═══");
        System.out.println();
        
        System.out.println("Test 1: Ohne Fehler:");
        try {
            System.out.println("  try: Code läuft...");
            int x = 10 / 2;  // Kein Fehler
        } catch (Exception e) {
            System.out.println("  catch: " + e.getMessage());
        } finally {
            System.out.println("  finally: Wird IMMER ausgeführt!");
        }
        
        System.out.println();
        System.out.println("Test 2: Mit Fehler:");
        try {
            System.out.println("  try: Code läuft...");
            int x = 10 / 0;  // 💥 ArithmeticException
        } catch (ArithmeticException e) {
            System.out.println("  catch: Division durch Null!");
        } finally {
            System.out.println("  finally: Wird IMMER ausgeführt!");
        }
        
        System.out.println();
        
        // ========================================
        // TEIL 4: Mehrere catch-Blöcke
        // ========================================
        
        System.out.println("═══ TEIL 4: Mehrere catch-Blöcke ═══");
        System.out.println();
        
        String[] inputs = {"42", "abc", null};
        
        for (String input : inputs) {
            System.out.println("Verarbeite: \"" + input + "\"");
            try {
                int zahl = Integer.parseInt(input);
                System.out.println("  → Zahl: " + zahl);
            } catch (NumberFormatException e) {
                System.out.println("  → ❌ Keine gültige Zahl!");
            } catch (NullPointerException e) {
                System.out.println("  → ❌ Input ist null!");
            }
        }
        
        System.out.println();
        
        // ========================================
        // TEIL 5: Eigene Exceptions
        // ========================================
        
        System.out.println("═══ TEIL 5: Eigene Exceptions ═══");
        System.out.println();
        
        Bankkonto konto1 = new Bankkonto("DE123", "Max Mustermann", 1000);
        Bankkonto konto2 = new Bankkonto("DE456", "Erika Musterfrau", 500);
        
        System.out.println();
        
        // Test 1: Normale Operationen
        System.out.println("→ Normale Einzahlung:");
        konto1.einzahlen(200);
        
        System.out.println();
        
        // Test 2: Unchecked Exception (UngueltigerBetragException)
        System.out.println("→ Ungültiger Betrag (unchecked):");
        try {
            konto1.einzahlen(-50);  // 💥 UngueltigerBetragException
        } catch (UngueltigerBetragException e) {
            System.out.println("  ❌ " + e.getMessage());
            System.out.println("  → Ungültiger Betrag war: " + e.getBetrag() + "€");
        }
        
        System.out.println();
        
        // Test 3: Checked Exception (KontoException)
        System.out.println("→ Nicht genug Guthaben (checked):");
        try {
            konto1.abheben(5000);  // 💥 KontoException
        } catch (KontoException e) {
            System.out.println("  ❌ " + e.getMessage());
        }
        
        System.out.println();
        
        // Test 4: Erfolgreiche Überweisung
        System.out.println("→ Erfolgreiche Überweisung:");
        try {
            konto1.ueberweisen(konto2, 300);
        } catch (KontoException e) {
            System.out.println("  ❌ Überweisung fehlgeschlagen: " + e.getMessage());
        }
        
        System.out.println();
        konto1.info();
        konto2.info();
        
        System.out.println();
        
        // ========================================
        // TEIL 6: Exception weiterleiten
        // ========================================
        
        System.out.println("═══ TEIL 6: Exception weiterleiten ═══");
        System.out.println();
        
        try {
            riskanteOperation();
        } catch (KontoException e) {
            System.out.println("❌ In main gefangen: " + e.getMessage());
        }
        
        System.out.println();
        
        // ========================================
        // TEIL 7: Best Practice Beispiel
        // ========================================
        
        System.out.println("═══ TEIL 7: Best Practice ═══");
        System.out.println();
        
        double betrag = sichereAbhebung(konto1, 100);
        System.out.println("Ergebnis: " + betrag + "€");
        
        betrag = sichereAbhebung(konto1, 999999);
        System.out.println("Ergebnis: " + betrag + "€");
        
        System.out.println();
        
        // ========================================
        // ABSCHLUSS
        // ========================================
        
        System.out.println("╔═══════════════════════════════════════════════════╗");
        System.out.println("║          🎉 KURS ABGESCHLOSSEN! 🎉                ║");
        System.out.println("╠═══════════════════════════════════════════════════╣");
        System.out.println("║                                                   ║");
        System.out.println("║  Du hast in 10 Tagen gelernt:                     ║");
        System.out.println("║                                                   ║");
        System.out.println("║  ✅ Klassen & Objekte                             ║");
        System.out.println("║  ✅ Attribute & Methoden                          ║");
        System.out.println("║  ✅ Kapselung & Sichtbarkeit                      ║");
        System.out.println("║  ✅ Konstruktoren                                 ║");
        System.out.println("║  ✅ Static & Konstanten                           ║");
        System.out.println("║  ✅ Vererbung & Polymorphie                       ║");
        System.out.println("║  ✅ Abstrakte Klassen                             ║");
        System.out.println("║  ✅ Typumwandlung & instanceof                    ║");
        System.out.println("║  ✅ Interfaces & Enums                            ║");
        System.out.println("║  ✅ Exception-Handling                            ║");
        System.out.println("║                                                   ║");
        System.out.println("║  Du bist bereit für die echte Welt! 🚀            ║");
        System.out.println("║                                                   ║");
        System.out.println("╚═══════════════════════════════════════════════════╝");
    }
    
    /**
     * Demonstriert Exception-Weiterleitung mit throws.
     */
    private static void riskanteOperation() throws KontoException {
        System.out.println("riskanteOperation() wird aufgerufen...");
        throw new KontoException("Simulierter Fehler in riskanteOperation()");
    }
    
    /**
     * Best Practice: Sichere Methode mit Exception-Handling.
     * Gibt 0 zurück bei Fehler, statt Exception weiterzugeben.
     */
    private static double sichereAbhebung(Bankkonto konto, double betrag) {
        System.out.println("Versuche " + betrag + "€ abzuheben...");
        try {
            konto.abheben(betrag);
            return betrag;
        } catch (KontoException e) {
            System.out.println("  ⚠️ Konnte nicht abheben: " + e.getMessage());
            return 0;
        } catch (UngueltigerBetragException e) {
            System.out.println("  ⚠️ Ungültiger Betrag: " + e.getBetrag());
            return 0;
        }
    }
}
