package de.javafleet.oop.model;

import de.javafleet.oop.exceptions.KontoException;
import de.javafleet.oop.exceptions.UngueltigerBetragException;

/**
 * Bankkonto — Demonstriert Exception-Handling.
 * 
 * Verwendet:
 * - UngueltigerBetragException (unchecked) für ungültige Beträge
 * - KontoException (checked) für Kontofehler
 * 
 * @author Nova Trent
 * @version 1.0
 */
public class Bankkonto {
    
    // ========================================
    // KONSTANTEN
    // ========================================
    
    public static final double MIN_EINZAHLUNG = 1.0;
    public static final double MAX_EINZELABHEBUNG = 5000.0;
    
    // ========================================
    // ATTRIBUTE
    // ========================================
    
    private final String kontonummer;
    private final String inhaber;
    private double kontostand;
    
    // ========================================
    // KONSTRUKTOR
    // ========================================
    
    /**
     * Erstellt ein neues Bankkonto.
     * 
     * @param kontonummer Die Kontonummer
     * @param inhaber Der Kontoinhaber
     * @param startguthaben Das Startguthaben (muss >= 0 sein)
     * @throws UngueltigerBetragException wenn Startguthaben negativ
     */
    public Bankkonto(String kontonummer, String inhaber, double startguthaben) {
        if (kontonummer == null || kontonummer.isBlank()) {
            throw new IllegalArgumentException("Kontonummer darf nicht leer sein!");
        }
        if (inhaber == null || inhaber.isBlank()) {
            throw new IllegalArgumentException("Inhaber darf nicht leer sein!");
        }
        if (startguthaben < 0) {
            throw new UngueltigerBetragException("Startguthaben darf nicht negativ sein", startguthaben);
        }
        
        this.kontonummer = kontonummer;
        this.inhaber = inhaber;
        this.kontostand = startguthaben;
        
        System.out.println("✓ Konto erstellt: " + kontonummer + " für " + inhaber);
        System.out.println("  Startguthaben: " + startguthaben + "€");
    }
    
    // ========================================
    // METHODEN MIT EXCEPTIONS
    // ========================================
    
    /**
     * Zahlt einen Betrag ein.
     * 
     * @param betrag Der einzuzahlende Betrag
     * @throws UngueltigerBetragException wenn Betrag ungültig
     */
    public void einzahlen(double betrag) {
        // Unchecked Exception für Programmierfehler
        if (betrag < MIN_EINZAHLUNG) {
            throw new UngueltigerBetragException(
                "Mindesteinzahlung: " + MIN_EINZAHLUNG + "€", betrag);
        }
        
        kontostand += betrag;
        System.out.println("💰 Eingezahlt: " + betrag + "€ | Neuer Stand: " + kontostand + "€");
    }
    
    /**
     * Hebt einen Betrag ab.
     * 
     * @param betrag Der abzuhebende Betrag
     * @throws UngueltigerBetragException wenn Betrag ungültig
     * @throws KontoException wenn nicht genug Guthaben
     */
    public void abheben(double betrag) throws KontoException {
        // Unchecked: Ungültiger Betrag
        if (betrag <= 0) {
            throw new UngueltigerBetragException("Betrag muss positiv sein", betrag);
        }
        if (betrag > MAX_EINZELABHEBUNG) {
            throw new UngueltigerBetragException(
                "Maximal " + MAX_EINZELABHEBUNG + "€ pro Abhebung", betrag);
        }
        
        // Checked: Geschäftslogik-Fehler
        if (betrag > kontostand) {
            throw new KontoException(
                "Nicht genug Guthaben! Verfügbar: " + kontostand + 
                "€, Angefordert: " + betrag + "€");
        }
        
        kontostand -= betrag;
        System.out.println("💸 Abgehoben: " + betrag + "€ | Neuer Stand: " + kontostand + "€");
    }
    
    /**
     * Überweist einen Betrag an ein anderes Konto.
     * 
     * @param ziel Das Zielkonto
     * @param betrag Der zu überweisende Betrag
     * @throws UngueltigerBetragException wenn Betrag ungültig
     * @throws KontoException wenn Überweisung fehlschlägt
     */
    public void ueberweisen(Bankkonto ziel, double betrag) throws KontoException {
        if (ziel == null) {
            throw new IllegalArgumentException("Zielkonto darf nicht null sein!");
        }
        if (ziel == this) {
            throw new IllegalArgumentException("Kann nicht an sich selbst überweisen!");
        }
        
        System.out.println("📤 Überweisung: " + betrag + "€ an " + ziel.getInhaber());
        
        // Erst abheben (kann KontoException werfen)
        this.abheben(betrag);
        
        // Dann einzahlen
        ziel.einzahlen(betrag);
        
        System.out.println("✅ Überweisung erfolgreich!");
    }
    
    // ========================================
    // INFO
    // ========================================
    
    public void info() {
        System.out.println("┌─────────────────────────────────────┐");
        System.out.println("│           KONTOINFORMATION          │");
        System.out.println("├─────────────────────────────────────┤");
        System.out.println("│ Konto:   " + String.format("%-26s", kontonummer) + " │");
        System.out.println("│ Inhaber: " + String.format("%-26s", inhaber) + " │");
        System.out.println("│ Stand:   " + String.format("%-26s", kontostand + "€") + " │");
        System.out.println("└─────────────────────────────────────┘");
    }
    
    // ========================================
    // GETTER
    // ========================================
    
    public String getKontonummer() { return kontonummer; }
    public String getInhaber() { return inhaber; }
    public double getKontostand() { return kontostand; }
}
