package de.javafleet.messaging.controller;

import de.javafleet.messaging.model.Order;
import de.javafleet.messaging.service.OrderConsumer;
import lombok.RequiredArgsConstructor;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.Map;

/**
 * REST-Controller für Consumer-Status.
 * 
 * Hier kannst du sehen, welche Bestellungen bereits verarbeitet wurden.
 * 
 * @author Java Fleet Systems Consulting
 */
@RestController
@RequestMapping("/api/consumer")
@RequiredArgsConstructor
public class ConsumerStatusController {

    private final OrderConsumer orderConsumer;

    /**
     * Gibt den aktuellen Status zurück.
     * 
     * HTTP: GET http://localhost:8082/api/consumer/status
     * 
     * @return Status mit Anzahl verarbeiteter Bestellungen
     */
    @GetMapping("/status")
    public Map<String, Object> getStatus() {
        return Map.of(
            "status", "RUNNING",
            "processedOrders", orderConsumer.getProcessedCount(),
            "message", "📥 Consumer is listening for messages..."
        );
    }
    
    /**
     * Gibt alle verarbeiteten Bestellungen zurück.
     * 
     * HTTP: GET http://localhost:8082/api/consumer/orders
     * 
     * @return Liste der Bestellungen
     */
    @GetMapping("/orders")
    public List<Order> getProcessedOrders() {
        return orderConsumer.getProcessedOrders();
    }
    
    /**
     * Health-Check für Monitoring.
     * 
     * HTTP: GET http://localhost:8082/api/consumer/health
     * 
     * @return "Consumer is running!"
     */
    @GetMapping("/health")
    public String health() {
        return "Consumer is running! 📥 Processed: " + orderConsumer.getProcessedCount();
    }
}
