package de.javafleet.messaging.service;

import de.javafleet.messaging.model.Order;
import lombok.extern.slf4j.Slf4j;
import org.springframework.jms.annotation.JmsListener;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * OrderConsumer - Empfängt und verarbeitet Bestellungen aus der Queue.
 * 
 * Was macht dieser Service?
 * Er lauscht auf die Queue und verarbeitet eingehende Bestellungen.
 * Das ist die "Lagerverwaltung" in unserem Szenario.
 * 
 * Wie funktioniert @JmsListener?
 * - Spring verbindet sich mit dem Broker
 * - Wenn eine Message in der Queue ist, wird die Methode aufgerufen
 * - Die Message wird automatisch in das richtige Objekt konvertiert
 * - Nach erfolgreicher Verarbeitung wird die Message bestätigt (ACK)
 * 
 * @author Java Fleet Systems Consulting
 */
@Service
@Slf4j
public class OrderConsumer {

    /**
     * Liste der verarbeiteten Bestellungen.
     * In Production würdest du eine Datenbank verwenden!
     */
    private final List<Order> processedOrders = Collections.synchronizedList(new ArrayList<>());

    /**
     * Empfängt Bestellungen aus der Queue.
     * 
     * @JmsListener macht die Magie:
     * - destination: Name der Queue (muss mit Producer übereinstimmen!)
     * - Spring ruft diese Methode automatisch auf
     * - Parameter wird aus Message konvertiert
     * 
     * Was passiert wenn die Verarbeitung fehlschlägt?
     * Die Message bleibt in der Queue und wird erneut zugestellt!
     * Das ist der große Vorteil von JMS - Guaranteed Delivery.
     * 
     * @param order Die empfangene Bestellung
     */
    @JmsListener(destination = "${app.queue.orders}")
    public void receiveOrder(Order order) {
        log.info("📥 ======================================");
        log.info("📥 ORDER RECEIVED!");
        log.info("📥 Order ID:  {}", order.getOrderId());
        log.info("📥 Customer:  {}", order.getCustomerName());
        log.info("📥 Product:   {}", order.getProduct());
        log.info("📥 Quantity:  {}", order.getQuantity());
        log.info("📥 Time:      {}", order.getOrderTime());
        log.info("📥 ======================================");
        
        // Simuliere Verarbeitung
        processOrder(order);
    }
    
    /**
     * Empfängt einfache Text-Messages.
     * 
     * Beachte: Gleiche Queue, anderer Typ!
     * ActiveMQ unterscheidet automatisch zwischen Order und String.
     * 
     * @param message Die empfangene Text-Nachricht
     */
    @JmsListener(destination = "${app.queue.orders}")
    public void receiveMessage(String message) {
        log.info("📥 Text message received: {}", message);
    }
    
    /**
     * Verarbeitet eine Bestellung.
     * 
     * In einem echten System würde hier:
     * - Lagerbestand geprüft
     * - Bestellung in DB gespeichert
     * - Versand angestoßen
     * - Email an Kunden gesendet
     * 
     * @param order Die zu verarbeitende Bestellung
     */
    private void processOrder(Order order) {
        log.info("⚙️ Processing order {}...", order.getOrderId());
        
        // Simuliere Verarbeitungszeit
        try {
            Thread.sleep(1000);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        
        // Status aktualisieren
        order.setStatus("COMPLETED");
        
        // Zur Liste hinzufügen
        processedOrders.add(order);
        
        log.info("✅ Order {} completed! Total processed: {}", 
            order.getOrderId(), processedOrders.size());
    }
    
    /**
     * Gibt alle verarbeiteten Bestellungen zurück.
     * Für den Status-Endpoint.
     * 
     * @return Liste der verarbeiteten Bestellungen
     */
    public List<Order> getProcessedOrders() {
        return new ArrayList<>(processedOrders);
    }
    
    /**
     * Gibt die Anzahl verarbeiteter Bestellungen zurück.
     * 
     * @return Anzahl
     */
    public int getProcessedCount() {
        return processedOrders.size();
    }
}
