package de.javafleet.messaging.controller;

import de.javafleet.messaging.model.Order;
import de.javafleet.messaging.service.OrderProducer;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.Map;

/**
 * REST-Controller für Bestellungen.
 * 
 * Dieser Controller nimmt Bestellungen entgegen und
 * schickt sie über den OrderProducer an die Message Queue.
 * 
 * Teste mit:
 * curl "http://localhost:8081/api/orders/send?product=Laptop&customer=Max&quantity=2"
 * 
 * @author Java Fleet Systems Consulting
 */
@RestController
@RequestMapping("/api/orders")
@RequiredArgsConstructor
@Slf4j
public class OrderController {

    private final OrderProducer orderProducer;

    /**
     * Sendet eine neue Bestellung an die Queue.
     * 
     * HTTP: GET /api/orders/send?product=X&customer=Y&quantity=Z
     * 
     * Warum GET statt POST?
     * Für dieses Tutorial ist GET einfacher zu testen (im Browser).
     * In Production würdest du POST mit JSON-Body verwenden!
     * 
     * @param product Produktname (required)
     * @param customer Kundenname (default: "TestKunde")
     * @param quantity Menge (default: 1)
     * @return Bestätigung mit Order-Details
     */
    @GetMapping("/send")
    public ResponseEntity<Map<String, Object>> sendOrder(
            @RequestParam String product,
            @RequestParam(defaultValue = "TestKunde") String customer,
            @RequestParam(defaultValue = "1") int quantity) {
        
        log.info("📝 New order received: {} x {} for {}", quantity, product, customer);
        
        // Bestellung erstellen
        Order order = Order.create(customer, product, quantity);
        
        // An Queue senden
        orderProducer.sendOrder(order);
        
        // Response zurückgeben
        return ResponseEntity.ok(Map.of(
            "status", "QUEUED",
            "orderId", order.getOrderId(),
            "message", "Order sent to queue! Consumer will process it.",
            "order", order
        ));
    }
    
    /**
     * Sendet eine einfache Text-Nachricht.
     * 
     * HTTP: GET /api/orders/message?text=Hello
     * 
     * @param text Die zu sendende Nachricht
     * @return Bestätigung
     */
    @GetMapping("/message")
    public ResponseEntity<Map<String, String>> sendMessage(
            @RequestParam String text) {
        
        orderProducer.sendMessage(text);
        
        return ResponseEntity.ok(Map.of(
            "status", "SENT",
            "message", text
        ));
    }
    
    /**
     * Health-Check Endpoint.
     * 
     * HTTP: GET /api/orders/health
     * 
     * @return "Producer is running!"
     */
    @GetMapping("/health")
    public String health() {
        return "Producer is running! 📤";
    }
}
