package de.javafleet.messaging.model;

import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;

import java.io.Serializable;
import java.time.LocalDateTime;
import java.util.UUID;

/**
 * Order - Repräsentiert eine Bestellung im Webshop.
 * 
 * Warum Serializable?
 * Damit die Bestellung als Message über den Broker geschickt werden kann,
 * muss sie serialisierbar sein. ActiveMQ wandelt das Objekt in Bytes um,
 * schickt es über das Netzwerk, und der Consumer baut es wieder zusammen.
 * 
 * Real-World-Szenario:
 * - Kunde bestellt im Webshop (Producer)
 * - Bestellung geht in die Queue
 * - Lagerverwaltung verarbeitet sie (Consumer)
 * 
 * @author Java Fleet Systems Consulting
 */
@Data
@NoArgsConstructor
@AllArgsConstructor
public class Order implements Serializable {
    
    private static final long serialVersionUID = 1L;
    
    /**
     * Eindeutige Bestell-ID.
     * UUID garantiert Einzigartigkeit auch über mehrere Systeme hinweg.
     */
    private String orderId;
    
    /**
     * Name des Kunden.
     */
    private String customerName;
    
    /**
     * Bestelltes Produkt.
     */
    private String product;
    
    /**
     * Bestellmenge.
     */
    private int quantity;
    
    /**
     * Zeitstempel der Bestellung.
     */
    private LocalDateTime orderTime;
    
    /**
     * Status der Bestellung.
     * PENDING → PROCESSING → COMPLETED
     */
    private String status;

    /**
     * Factory-Methode für neue Bestellungen.
     * Generiert automatisch ID und Zeitstempel.
     * 
     * @param customerName Name des Kunden
     * @param product Bestelltes Produkt
     * @param quantity Menge
     * @return Neue Order mit generierter ID
     */
    public static Order create(String customerName, String product, int quantity) {
        return new Order(
            UUID.randomUUID().toString().substring(0, 8),
            customerName,
            product,
            quantity,
            LocalDateTime.now(),
            "PENDING"
        );
    }
}
