package de.javafleet.messaging.service;

import de.javafleet.messaging.model.Order;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jms.core.JmsTemplate;
import org.springframework.stereotype.Service;

/**
 * OrderProducer - Sendet Bestellungen an die Message Queue.
 * 
 * Was macht dieser Service?
 * Er nimmt Bestellungen entgegen und schickt sie als Messages
 * an den ActiveMQ Broker. Von dort werden sie an Consumer verteilt.
 * 
 * Warum nicht direkt verarbeiten?
 * - Entkopplung: Webshop muss nicht warten bis Lager fertig ist
 * - Skalierbarkeit: Mehrere Consumer können parallel arbeiten
 * - Resilience: Wenn ein Consumer ausfällt, wartet die Message
 * 
 * @author Java Fleet Systems Consulting
 */
@Service
@Slf4j
public class OrderProducer {

    private final JmsTemplate jmsTemplate;
    private final String queueName;

    /**
     * Konstruktor mit Dependency Injection.
     * 
     * @param jmsTemplate Spring's JMS-Helfer für Message-Versand
     * @param queueName Name der Queue aus application.properties
     */
    public OrderProducer(
            JmsTemplate jmsTemplate, 
            @Value("${app.queue.orders}") String queueName) {
        this.jmsTemplate = jmsTemplate;
        this.queueName = queueName;
    }

    /**
     * Sendet eine Bestellung an die Queue.
     * 
     * Was passiert hier?
     * 1. Order wird serialisiert (in Bytes umgewandelt)
     * 2. Bytes werden an den Broker geschickt
     * 3. Broker speichert Message in der Queue
     * 4. Methode returned SOFORT (nicht blockierend!)
     * 
     * Der Consumer bekommt die Message, sobald er bereit ist.
     * 
     * @param order Die zu sendende Bestellung
     */
    public void sendOrder(Order order) {
        log.info("📤 Sending order to queue '{}': {}", queueName, order.getOrderId());
        
        try {
            // Das ist der Kern: convertAndSend
            // - convert: Wandelt Order in Message um
            // - send: Schickt an die Queue
            jmsTemplate.convertAndSend(queueName, order);
            
            log.info("✅ Order {} sent successfully!", order.getOrderId());
            
        } catch (Exception e) {
            // Wichtig: Broker nicht erreichbar?
            log.error("❌ Failed to send order {}: {}", order.getOrderId(), e.getMessage());
            throw new RuntimeException("Could not send order to queue", e);
        }
    }
    
    /**
     * Sendet eine einfache Text-Message.
     * Gut für einfache Benachrichtigungen.
     * 
     * @param message Die zu sendende Nachricht
     */
    public void sendMessage(String message) {
        log.info("📤 Sending text message to queue '{}'", queueName);
        jmsTemplate.convertAndSend(queueName, message);
        log.info("✅ Message sent!");
    }
}
