package de.javafleet.messaging.config;

import org.apache.activemq.broker.BrokerService;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

/**
 * Konfiguration für den eingebetteten ActiveMQ Broker.
 * 
 * Was macht diese Klasse?
 * Sie erstellt und konfiguriert einen ActiveMQ Message Broker,
 * der als zentrale Vermittlungsstelle für alle Messages dient.
 * 
 * Warum embedded?
 * Für Development und kleine Anwendungen ist ein eingebetteter Broker
 * perfekt - keine separate Installation nötig!
 * In Production würdest du einen standalone Broker verwenden.
 * 
 * @author Java Fleet Systems Consulting
 */
@Configuration
public class BrokerConfig {

    @Value("${broker.url:tcp://localhost:61616}")
    private String brokerUrl;
    
    @Value("${broker.data.directory:./data/activemq}")
    private String dataDirectory;
    
    @Value("${broker.persistent:true}")
    private boolean persistent;

    /**
     * Erstellt den BrokerService Bean.
     * 
     * Der BrokerService ist das Herzstück - er:
     * - Empfängt Messages von Producern
     * - Speichert sie in Queues
     * - Liefert sie an Consumer aus
     * 
     * @return Konfigurierter BrokerService
     * @throws Exception wenn der Broker nicht gestartet werden kann
     */
    @Bean
    public BrokerService brokerService() throws Exception {
        BrokerService broker = new BrokerService();
        
        // TCP-Connector für externe Verbindungen
        // Producer und Consumer verbinden sich über diese URL
        broker.addConnector(brokerUrl);
        
        // Persistenz aktivieren = Messages überleben Neustarts!
        // Das ist der große Vorteil gegenüber @Async
        broker.setPersistent(persistent);
        
        // Datenverzeichnis für persistente Messages
        // Hier speichert ActiveMQ die KahaDB-Datenbank
        broker.setDataDirectory(dataDirectory);
        
        // Broker-Name für Logging und Debugging
        broker.setBrokerName("JavaFleetBroker");
        
        return broker;
    }
}
