package de.javafleet.messaging.controller;

import org.apache.activemq.broker.BrokerService;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.Map;

/**
 * REST-Controller für Broker-Status und Health-Checks.
 * 
 * Warum dieser Controller?
 * In Production willst du wissen, ob dein Broker läuft.
 * Dieser Endpoint gibt dir diese Info auf einen Blick.
 * 
 * @author Java Fleet Systems Consulting
 */
@RestController
@RequestMapping("/api/broker")
public class BrokerStatusController {

    private final BrokerService brokerService;

    public BrokerStatusController(BrokerService brokerService) {
        this.brokerService = brokerService;
    }

    /**
     * Gibt den aktuellen Broker-Status zurück.
     * 
     * Aufruf: GET http://localhost:8080/api/broker/status
     * 
     * @return Map mit Status-Informationen
     */
    @GetMapping("/status")
    public Map<String, Object> getStatus() {
        return Map.of(
            "brokerName", brokerService.getBrokerName(),
            "started", brokerService.isStarted(),
            "persistent", brokerService.isPersistent(),
            "dataDirectory", brokerService.getDataDirectoryFile().getAbsolutePath(),
            "message", "🚀 Broker läuft und wartet auf Messages!"
        );
    }
    
    /**
     * Einfacher Health-Check für Load Balancer / Monitoring.
     * 
     * Aufruf: GET http://localhost:8080/api/broker/health
     * 
     * @return "OK" wenn Broker läuft
     */
    @GetMapping("/health")
    public String health() {
        return brokerService.isStarted() ? "OK" : "STARTING";
    }
}
