package de.javafleet.messaging.controller;

import de.javafleet.messaging.service.EmailService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.Map;

/**
 * REST-Controller für Email-Versand.
 * 
 * Teste mit:
 * - Einfache Email: curl "http://localhost:8083/api/email/send?to=test@test.de"
 * - HTML Email:     curl "http://localhost:8083/api/email/html?to=test@test.de&name=Max"
 * - Async Email:    curl "http://localhost:8083/api/email/async?to=test@test.de&name=Max"
 * 
 * @author Java Fleet Systems Consulting
 */
@RestController
@RequestMapping("/api/email")
@RequiredArgsConstructor
@Slf4j
public class EmailController {

    private final EmailService emailService;

    /**
     * Sendet eine einfache Text-Email.
     * 
     * HTTP: GET /api/email/send?to=X&subject=Y&text=Z
     * 
     * @param to Empfänger (required)
     * @param subject Betreff (default: "Test Email")
     * @param text Nachricht (default: "Hello from Java Fleet!")
     * @return Bestätigung
     */
    @GetMapping("/send")
    public ResponseEntity<Map<String, String>> sendSimpleEmail(
            @RequestParam String to,
            @RequestParam(defaultValue = "Test Email von Java Fleet") String subject,
            @RequestParam(defaultValue = "Hello from Java Fleet! 🚀") String text) {
        
        long start = System.currentTimeMillis();
        
        emailService.sendSimpleEmail(to, subject, text);
        
        long duration = System.currentTimeMillis() - start;
        
        return ResponseEntity.ok(Map.of(
            "status", "SENT",
            "to", to,
            "duration", duration + "ms",
            "message", "Check MailDev at http://localhost:1080"
        ));
    }

    /**
     * Sendet eine HTML-Email mit Template.
     * 
     * HTTP: GET /api/email/html?to=X&name=Y
     * 
     * @param to Empfänger (required)
     * @param name Name für das Template (default: "Entwickler")
     * @return Bestätigung
     */
    @GetMapping("/html")
    public ResponseEntity<Map<String, String>> sendHtmlEmail(
            @RequestParam String to,
            @RequestParam(defaultValue = "Entwickler") String name) {
        
        long start = System.currentTimeMillis();
        
        emailService.sendHtmlEmail(to, "Willkommen bei Java Fleet! 🚀", name, to);
        
        long duration = System.currentTimeMillis() - start;
        
        return ResponseEntity.ok(Map.of(
            "status", "SENT",
            "to", to,
            "duration", duration + "ms",
            "message", "HTML Email sent! Check MailDev at http://localhost:1080"
        ));
    }

    /**
     * Sendet eine Email ASYNCHRON.
     * 
     * HTTP: GET /api/email/async?to=X&name=Y
     * 
     * Beachte die Response-Zeit! Sie ist VIEL schneller,
     * weil der eigentliche Versand im Hintergrund läuft.
     * 
     * @param to Empfänger (required)
     * @param name Name für das Template (default: "Entwickler")
     * @return Bestätigung (sofort, auch wenn Email noch sendet)
     */
    @GetMapping("/async")
    public ResponseEntity<Map<String, String>> sendAsyncEmail(
            @RequestParam String to,
            @RequestParam(defaultValue = "Entwickler") String name) {
        
        long start = System.currentTimeMillis();
        
        // Diese Methode returned SOFORT!
        emailService.sendWelcomeEmailAsync(to, "Willkommen bei Java Fleet! 🚀", name);
        
        long duration = System.currentTimeMillis() - start;
        
        return ResponseEntity.ok(Map.of(
            "status", "QUEUED",
            "to", to,
            "duration", duration + "ms",
            "message", "Email wird im Hintergrund gesendet! Check MailDev in ~2 Sekunden."
        ));
    }

    /**
     * Vergleicht sync vs async Response-Zeit.
     * 
     * HTTP: GET /api/email/compare?to=X
     * 
     * Zeigt den Performance-Unterschied!
     * 
     * @param to Empfänger
     * @return Vergleich der Zeiten
     */
    @GetMapping("/compare")
    public ResponseEntity<Map<String, Object>> comparePerformance(
            @RequestParam String to) {
        
        // Sync messen
        long syncStart = System.currentTimeMillis();
        emailService.sendSimpleEmail(to, "Sync Test", "This was sent synchronously");
        long syncDuration = System.currentTimeMillis() - syncStart;
        
        // Async messen
        long asyncStart = System.currentTimeMillis();
        emailService.sendWelcomeEmailAsync(to, "Async Test", "User");
        long asyncDuration = System.currentTimeMillis() - asyncStart;
        
        return ResponseEntity.ok(Map.of(
            "syncDuration", syncDuration + "ms",
            "asyncDuration", asyncDuration + "ms",
            "speedup", (syncDuration / (double) Math.max(asyncDuration, 1)) + "x faster",
            "explanation", "Async returned immediately, email sends in background!"
        ));
    }

    /**
     * Health-Check.
     */
    @GetMapping("/health")
    public String health() {
        return "Email Service is running! 📧";
    }
}
