package de.javafleet.messaging.service;

import jakarta.mail.MessagingException;
import jakarta.mail.internet.MimeMessage;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.mail.SimpleMailMessage;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.thymeleaf.context.Context;
import org.thymeleaf.spring6.SpringTemplateEngine;

/**
 * EmailService - Versendet Emails über Spring Boot Mail.
 * 
 * Dieser Service bietet drei Arten von Email-Versand:
 * 1. Einfache Text-Emails (synchron)
 * 2. HTML-Emails mit Thymeleaf Templates (synchron)
 * 3. Asynchroner Email-Versand für bessere Performance
 * 
 * Warum asynchron?
 * Email-Versand kann 2-5 Sekunden dauern. Wenn ein User sich
 * registriert, soll er nicht so lange warten. Mit @Async
 * läuft der Versand im Hintergrund!
 * 
 * @author Java Fleet Systems Consulting
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class EmailService {

    private final JavaMailSender mailSender;
    private final SpringTemplateEngine templateEngine;
    
    @Value("${spring.mail.from:noreply@java-developer.online}")
    private String fromEmail;

    /**
     * Sendet eine einfache Text-Email.
     * 
     * Wann verwenden?
     * - Einfache Benachrichtigungen
     * - System-Alerts
     * - Schnelle Tests
     * 
     * @param to Empfänger-Adresse
     * @param subject Betreff
     * @param text Nachrichtentext
     */
    public void sendSimpleEmail(String to, String subject, String text) {
        log.info("📧 Sending simple email to: {}", to);
        
        SimpleMailMessage message = new SimpleMailMessage();
        message.setFrom(fromEmail);
        message.setTo(to);
        message.setSubject(subject);
        message.setText(text);
        
        mailSender.send(message);
        
        log.info("✅ Simple email sent to: {}", to);
    }

    /**
     * Sendet eine HTML-Email mit Thymeleaf Template.
     * 
     * Wann verwenden?
     * - Willkommens-Emails
     * - Bestellbestätigungen
     * - Marketing-Emails
     * 
     * @param to Empfänger-Adresse
     * @param subject Betreff
     * @param username Name des Empfängers (für Template)
     * @param email Email-Adresse (für Template)
     */
    public void sendHtmlEmail(String to, String subject, String username, String email) {
        log.info("📧 Sending HTML email to: {}", to);
        
        try {
            // Thymeleaf Context mit Variablen befüllen
            Context context = new Context();
            context.setVariable("username", username);
            context.setVariable("email", email);
            
            // Template verarbeiten → HTML-String
            String htmlContent = templateEngine.process("email/welcome", context);
            
            // MimeMessage für HTML erstellen
            MimeMessage mimeMessage = mailSender.createMimeMessage();
            MimeMessageHelper helper = new MimeMessageHelper(mimeMessage, true, "UTF-8");
            
            helper.setFrom(fromEmail);
            helper.setTo(to);
            helper.setSubject(subject);
            helper.setText(htmlContent, true);  // true = ist HTML!
            
            mailSender.send(mimeMessage);
            
            log.info("✅ HTML email sent to: {}", to);
            
        } catch (MessagingException e) {
            log.error("❌ Failed to send HTML email to: {}", to, e);
            throw new RuntimeException("Email sending failed", e);
        }
    }

    /**
     * Sendet eine Email ASYNCHRON.
     * 
     * Das ist der Game-Changer!
     * 
     * @Async bewirkt:
     * - Methode läuft in separatem Thread
     * - Aufrufer wartet NICHT
     * - Perfekt für User-Registrierungen
     * 
     * WICHTIG: @EnableAsync muss in der Main-Klasse sein!
     * 
     * @param to Empfänger-Adresse
     * @param subject Betreff
     * @param username Name des Empfängers
     */
    @Async
    public void sendWelcomeEmailAsync(String to, String subject, String username) {
        log.info("📧 [ASYNC] Starting async email to: {}", to);
        
        try {
            // Simuliere: Email-Versand dauert eine Weile
            Thread.sleep(2000);
            
            sendHtmlEmail(to, subject, username, to);
            
            log.info("✅ [ASYNC] Email sent to: {}", to);
            
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            log.error("❌ [ASYNC] Interrupted while sending email");
        } catch (Exception e) {
            // Wichtig: Exceptions in @Async werden nicht propagiert!
            // Hier musst du sie selbst loggen/behandeln
            log.error("❌ [ASYNC] Failed to send email to: {}", to, e);
        }
    }
}
