# Greeting Spring Boot Starter

[![Java](https://img.shields.io/badge/Java-17+-blue.svg)](https://openjdk.org/)
[![Spring Boot](https://img.shields.io/badge/Spring%20Boot-3.2.x-brightgreen.svg)](https://spring.io/projects/spring-boot)
[![License](https://img.shields.io/badge/License-MIT-yellow.svg)](LICENSE)

Ein Custom Spring Boot Starter der einen konfigurierbaren `GreetingService` bereitstellt.

**Begleitprojekt zum Blogbeitrag:**  
👉 [Tag 1: Auto-Configuration & Custom Starter](https://www.java-developer.online/tag-1-auto-configuration-custom-starter/)

---

## 📦 Projekt-Struktur

```
greeting-spring-boot-starter-project/
├── pom.xml                                  # Parent POM (Multi-Module)
├── greeting-spring-boot-starter/            # Der Starter
│   ├── pom.xml
│   └── src/
│       ├── main/java/com/javafleet/greeting/
│       │   ├── GreetingService.java         # Interface
│       │   ├── DefaultGreetingService.java  # Implementierung
│       │   ├── GreetingProperties.java      # Configuration Properties
│       │   └── GreetingAutoConfiguration.java
│       ├── main/resources/META-INF/spring/
│       │   └── org.springframework.boot.autoconfigure.AutoConfiguration.imports
│       └── test/java/com/javafleet/greeting/
│           ├── GreetingPropertiesTest.java
│           ├── DefaultGreetingServiceTest.java
│           └── GreetingAutoConfigurationTest.java
│
└── greeting-spring-boot-starter-demo/       # Demo-Anwendung
    ├── pom.xml
    └── src/
        ├── main/java/com/javafleet/demo/
        │   ├── DemoApplication.java
        │   └── GreetingController.java
        ├── main/resources/
        │   └── application.properties
        └── test/java/com/javafleet/demo/
            ├── DemoApplicationTest.java
            └── GreetingControllerTest.java
```

---

## 🚀 Schnellstart

### 1. Projekt bauen

```bash
# Komplettes Projekt bauen und Tests ausführen
mvn clean install

# Nur Starter bauen
cd greeting-spring-boot-starter
mvn clean install
```

### 2. Demo starten

```bash
cd greeting-spring-boot-starter-demo
mvn spring-boot:run
```

### 3. Testen

```bash
# REST-Endpunkt testen
curl http://localhost:8080/greet/Franz
# Output: Hello, Franz!

curl http://localhost:8080/greet
# Output: Hello, World!
```

---

## ⚙️ Konfiguration

Alle Properties können in `application.properties` gesetzt werden:

```properties
# Basis-Konfiguration
greeting.message=Guten Tag
greeting.uppercase=true
greeting.prefix=[VIP]
greeting.enabled=true

# Format-Optionen (Nested)
greeting.format.emoji=👋
greeting.format.separator=, 
greeting.format.suffix=!
```

### Beispiel-Outputs

| Konfiguration | Output für `greet("Franz")` |
|---------------|----------------------------|
| Defaults | `Hello, Franz!` |
| `greeting.message=Hi` | `Hi, Franz!` |
| `greeting.uppercase=true` | `HELLO, FRANZ!` |
| `greeting.prefix=[VIP]` | `[VIP] Hello, Franz!` |
| `greeting.format.emoji=👋` | `👋 Hello, Franz! 👋` |
| Alle kombiniert | `👋 [VIP] GUTEN TAG, FRANZ! 👋` |

---

## 🧪 Tests

Das Projekt enthält umfangreiche Unit- und Integration-Tests:

```bash
# Alle Tests ausführen
mvn test

# Nur Starter-Tests
cd greeting-spring-boot-starter
mvn test

# Nur Demo-Tests
cd greeting-spring-boot-starter-demo
mvn test
```

### Test-Abdeckung

- **GreetingPropertiesTest**: Default-Werte, Getter/Setter
- **DefaultGreetingServiceTest**: Alle Kombinationen, Edge Cases
- **GreetingAutoConfigurationTest**: ApplicationContextRunner-Tests
  - Bean-Erstellung
  - Property Binding
  - @ConditionalOnProperty
  - @ConditionalOnMissingBean (Überschreibbarkeit)
- **GreetingControllerTest**: REST-Endpunkte mit MockMvc

---

## 🔧 Eigene Implementierung überschreiben

Dank `@ConditionalOnMissingBean` kannst du eine eigene Implementierung bereitstellen:

```java
@Configuration
public class MyConfig {
    
    @Bean
    public GreetingService greetingService() {
        return name -> "🎉 Custom Greeting: " + name + " 🎉";
    }
}
```

Die `DefaultGreetingService`-Bean wird dann **nicht** erstellt!

---

## 🐛 Debug-Modus

Um zu sehen welche Auto-Configurations geladen werden:

```properties
debug=true
```

Dann erscheint in der Console der **Conditions Evaluation Report**.

---

## 📚 Weiterführende Links

- [Spring Boot Reference: Creating Your Own Starter](https://docs.spring.io/spring-boot/docs/current/reference/html/features.html#features.developing-auto-configuration.custom-starter)
- [Spring Boot Reference: Configuration Properties](https://docs.spring.io/spring-boot/docs/current/reference/html/features.html#features.external-config.typesafe-configuration-properties)
- [ApplicationContextRunner Documentation](https://docs.spring.io/spring-boot/docs/current/api/org/springframework/boot/test/context/runner/ApplicationContextRunner.html)

---

## 📄 Lizenz

MIT License - siehe [LICENSE](LICENSE)

---

**Java Fleet Systems Consulting**  
*Keep coding, keep learning!* 💙
