package com.javafleet.demo;

import com.javafleet.greeting.GreetingService;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

/**
 * REST-Controller für die Greeting-Demo.
 * 
 * <h2>Endpunkte:</h2>
 * <ul>
 *   <li>{@code GET /greet/{name}} - Begrüßung für einen Namen</li>
 *   <li>{@code GET /greet} - Begrüßung für "World" (Default)</li>
 * </ul>
 * 
 * <h2>Beispiele:</h2>
 * <pre>
 * curl http://localhost:8080/greet/Franz
 * # Output: Hello, Franz!
 * 
 * curl http://localhost:8080/greet
 * # Output: Hello, World!
 * </pre>
 */
@RestController
@RequestMapping("/greet")
public class GreetingController {

    private final GreetingService greetingService;

    /**
     * Constructor Injection - der GreetingService wird automatisch
     * durch unseren Starter bereitgestellt!
     */
    public GreetingController(GreetingService greetingService) {
        this.greetingService = greetingService;
    }

    /**
     * Begrüßung mit spezifischem Namen.
     * 
     * @param name Der zu begrüßende Name
     * @return Die Begrüßung als Plain Text
     */
    @GetMapping("/{name}")
    public ResponseEntity<String> greet(@PathVariable String name) {
        String greeting = greetingService.greet(name);
        return ResponseEntity.ok(greeting);
    }

    /**
     * Default-Begrüßung für "World".
     * 
     * @return Die Begrüßung als Plain Text
     */
    @GetMapping
    public ResponseEntity<String> greetWorld() {
        return greet("World");
    }
}
