package com.javafleet.greeting;

/**
 * Standard-Implementierung des {@link GreetingService}.
 * 
 * <p>Diese Klasse wird automatisch von der {@link GreetingAutoConfiguration}
 * als Spring Bean erstellt, sofern keine andere Implementierung vorhanden ist.</p>
 * 
 * <p>Die Begrüßung wird anhand der konfigurierten {@link GreetingProperties}
 * formatiert. Folgende Features werden unterstützt:</p>
 * 
 * <ul>
 *   <li><b>Message:</b> Anpassbare Begrüßung (z.B. "Hello", "Guten Tag")</li>
 *   <li><b>Prefix:</b> Optionaler Prefix (z.B. "[VIP]")</li>
 *   <li><b>Uppercase:</b> Ausgabe in Großbuchstaben</li>
 *   <li><b>Emoji:</b> Optionales Emoji am Anfang und Ende</li>
 *   <li><b>Separator:</b> Anpassbarer Separator zwischen Message und Name</li>
 * </ul>
 * 
 * <h2>Beispiel:</h2>
 * <pre>
 * greeting.message=Welcome
 * greeting.prefix=[PREMIUM]
 * greeting.uppercase=true
 * greeting.format.emoji=🎉
 * 
 * // Ergebnis für greet("Franz"):
 * // "🎉 [PREMIUM] WELCOME, FRANZ! 🎉"
 * </pre>
 * 
 * @author Java Fleet Systems Consulting
 * @since 1.0.0
 */
public class DefaultGreetingService implements GreetingService {

    private final GreetingProperties properties;

    /**
     * Konstruktor mit Dependency Injection.
     * 
     * <p>Constructor Injection ist Best Practice in Spring:
     * <ul>
     *   <li>Immutability: Properties können final sein</li>
     *   <li>Testbarkeit: Einfaches Mocken in Unit Tests</li>
     *   <li>Required Dependencies: Compiler garantiert Vollständigkeit</li>
     * </ul>
     * </p>
     * 
     * @param properties Die Konfiguration aus application.properties
     * @throws IllegalArgumentException wenn properties null ist
     */
    public DefaultGreetingService(GreetingProperties properties) {
        if (properties == null) {
            throw new IllegalArgumentException("GreetingProperties must not be null");
        }
        this.properties = properties;
    }

    /**
     * {@inheritDoc}
     * 
     * <p>Baut die Begrüßung in folgender Reihenfolge:</p>
     * <ol>
     *   <li>Prefix (falls konfiguriert)</li>
     *   <li>Message</li>
     *   <li>Separator</li>
     *   <li>Name</li>
     *   <li>Suffix</li>
     *   <li>Uppercase (falls aktiviert)</li>
     *   <li>Emoji (falls konfiguriert, am Anfang und Ende)</li>
     * </ol>
     */
    @Override
    public String greet(String name) {
        if (name == null || name.trim().isEmpty()) {
            throw new IllegalArgumentException("Name must not be null or empty");
        }

        StringBuilder greeting = new StringBuilder();
        
        // 1. Prefix hinzufügen (falls konfiguriert)
        String prefix = properties.getPrefix();
        if (prefix != null && !prefix.isEmpty()) {
            greeting.append(prefix).append(" ");
        }
        
        // 2. Message hinzufügen
        greeting.append(properties.getMessage());
        
        // 3. Separator und Name
        greeting.append(properties.getFormat().getSeparator());
        greeting.append(name.trim());
        
        // 4. Suffix
        greeting.append(properties.getFormat().getSuffix());
        
        // 5. Uppercase (vor Emoji, damit Emoji nicht affected wird)
        String result = greeting.toString();
        if (properties.isUppercase()) {
            result = result.toUpperCase();
        }
        
        // 6. Emoji hinzufügen (falls konfiguriert)
        String emoji = properties.getFormat().getEmoji();
        if (emoji != null && !emoji.isEmpty()) {
            result = emoji + " " + result + " " + emoji;
        }
        
        return result;
    }

    /**
     * Gibt die aktuellen Properties zurück.
     * 
     * <p>Nützlich für Debugging und Logging.</p>
     * 
     * @return Die verwendeten Properties
     */
    public GreetingProperties getProperties() {
        return properties;
    }
}
