package com.javafleet.greeting;

import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;

/**
 * Auto-Configuration für den Greeting Spring Boot Starter.
 * 
 * <p>Diese Klasse wird automatisch von Spring Boot geladen wenn:
 * <ul>
 *   <li>Der Starter im Classpath ist</li>
 *   <li>Die {@link GreetingService} Klasse verfügbar ist</li>
 *   <li>{@code greeting.enabled} nicht explizit auf {@code false} gesetzt ist</li>
 * </ul>
 * </p>
 * 
 * <h2>Conditional Annotations erklärt:</h2>
 * <ul>
 *   <li><b>@AutoConfiguration:</b> Markiert diese Klasse als Auto-Configuration (Spring Boot 3+)</li>
 *   <li><b>@ConditionalOnClass:</b> Lädt nur wenn bestimmte Klassen im Classpath sind</li>
 *   <li><b>@ConditionalOnMissingBean:</b> Erstellt Bean nur wenn keiner existiert (Überschreibbarkeit!)</li>
 *   <li><b>@ConditionalOnProperty:</b> Abhängig von Property-Werten in application.properties</li>
 * </ul>
 * 
 * <h2>Überschreibbarkeit:</h2>
 * <p>Dank {@code @ConditionalOnMissingBean} kann der User eine eigene
 * {@link GreetingService}-Implementierung bereitstellen:</p>
 * 
 * <pre>{@code
 * @Configuration
 * public class MyConfig {
 *     @Bean
 *     public GreetingService greetingService() {
 *         return new MyCustomGreetingService();
 *     }
 * }
 * }</pre>
 * 
 * <p>In diesem Fall wird {@link DefaultGreetingService} NICHT erstellt!</p>
 * 
 * @author Java Fleet Systems Consulting
 * @since 1.0.0
 * @see GreetingService
 * @see GreetingProperties
 * @see DefaultGreetingService
 */
@AutoConfiguration
@ConditionalOnClass(GreetingService.class)
@EnableConfigurationProperties(GreetingProperties.class)
public class GreetingAutoConfiguration {

    /**
     * Erstellt den {@link GreetingService} Bean.
     * 
     * <p>Dieser Bean wird nur erstellt wenn:</p>
     * <ol>
     *   <li>Kein anderer {@code GreetingService} Bean existiert</li>
     *   <li>{@code greeting.enabled} ist {@code true} oder nicht gesetzt</li>
     * </ol>
     * 
     * <p><b>Wichtig:</b> {@code matchIfMissing = true} folgt dem Spring Boot
     * Prinzip "Convention over Configuration" – der Starter funktioniert
     * out-of-the-box ohne explizite Konfiguration.</p>
     * 
     * @param properties Die automatisch gebundenen Configuration Properties
     * @return Eine neue Instanz von {@link DefaultGreetingService}
     */
    @Bean
    @ConditionalOnMissingBean(GreetingService.class)
    @ConditionalOnProperty(
        prefix = "greeting",
        name = "enabled",
        havingValue = "true",
        matchIfMissing = true  // Wichtig: Default ist aktiviert!
    )
    public GreetingService greetingService(GreetingProperties properties) {
        return new DefaultGreetingService(properties);
    }
}
