package com.javafleet.greeting;

import org.springframework.boot.context.properties.ConfigurationProperties;

/**
 * Configuration Properties für den Greeting Starter.
 * 
 * <p>Alle Properties mit dem Prefix {@code greeting} werden automatisch 
 * an diese Klasse gebunden. Spring Boot's relaxed binding erlaubt dabei
 * verschiedene Schreibweisen:</p>
 * 
 * <ul>
 *   <li>{@code greeting.message} (Standard)</li>
 *   <li>{@code greeting.MESSAGE} (Uppercase)</li>
 *   <li>{@code GREETING_MESSAGE} (Environment Variable)</li>
 * </ul>
 * 
 * <h2>Beispiel application.properties:</h2>
 * <pre>
 * greeting.message=Hello
 * greeting.uppercase=true
 * greeting.prefix=[VIP]
 * greeting.enabled=true
 * greeting.format.emoji=👋
 * greeting.format.separator=,
 * </pre>
 * 
 * @author Java Fleet Systems Consulting
 * @since 1.0.0
 */
@ConfigurationProperties(prefix = "greeting")
public class GreetingProperties {

    /**
     * Die Begrüßungs-Message.
     * Standard: "Hello"
     */
    private String message = "Hello";

    /**
     * Soll die Ausgabe in Großbuchstaben sein?
     * Standard: false
     */
    private boolean uppercase = false;

    /**
     * Optionaler Prefix vor der Message.
     * Wenn gesetzt, wird er der Begrüßung vorangestellt.
     */
    private String prefix = "";

    /**
     * Aktiviert/deaktiviert den Greeting Service.
     * Wenn false, wird keine Bean erstellt.
     * Standard: true
     */
    private boolean enabled = true;

    /**
     * Format-Optionen für erweiterte Anpassung.
     */
    private Format format = new Format();

    // ==================== GETTER & SETTER ====================

    public String getMessage() {
        return message;
    }

    public void setMessage(String message) {
        this.message = message;
    }

    public boolean isUppercase() {
        return uppercase;
    }

    public void setUppercase(boolean uppercase) {
        this.uppercase = uppercase;
    }

    public String getPrefix() {
        return prefix;
    }

    public void setPrefix(String prefix) {
        this.prefix = prefix;
    }

    public boolean isEnabled() {
        return enabled;
    }

    public void setEnabled(boolean enabled) {
        this.enabled = enabled;
    }

    public Format getFormat() {
        return format;
    }

    public void setFormat(Format format) {
        this.format = format;
    }

    // ==================== NESTED CLASS ====================

    /**
     * Nested Configuration für Format-Optionen.
     * 
     * <p>Konfigurierbar über {@code greeting.format.*}</p>
     */
    public static class Format {

        /**
         * Optionales Emoji das der Begrüßung hinzugefügt wird.
         * Beispiel: "👋" → "👋 Hello, World! 👋"
         */
        private String emoji = "";

        /**
         * Separator zwischen Message und Name.
         * Standard: ", " → "Hello, World!"
         */
        private String separator = ", ";

        /**
         * Suffix nach dem Namen.
         * Standard: "!" → "Hello, World!"
         */
        private String suffix = "!";

        public String getEmoji() {
            return emoji;
        }

        public void setEmoji(String emoji) {
            this.emoji = emoji;
        }

        public String getSeparator() {
            return separator;
        }

        public void setSeparator(String separator) {
            this.separator = separator;
        }

        public String getSuffix() {
            return suffix;
        }

        public void setSuffix(String suffix) {
            this.suffix = suffix;
        }

        @Override
        public String toString() {
            return "Format{" +
                    "emoji='" + emoji + '\'' +
                    ", separator='" + separator + '\'' +
                    ", suffix='" + suffix + '\'' +
                    '}';
        }
    }

    @Override
    public String toString() {
        return "GreetingProperties{" +
                "message='" + message + '\'' +
                ", uppercase=" + uppercase +
                ", prefix='" + prefix + '\'' +
                ", enabled=" + enabled +
                ", format=" + format +
                '}';
    }
}
