package com.javafleet.greeting;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Nested;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.NullAndEmptySource;
import org.junit.jupiter.params.provider.ValueSource;

import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.assertThatThrownBy;

/**
 * Unit Tests für {@link DefaultGreetingService}.
 * 
 * <p>Testet alle Kombinationen von Properties und Edge Cases.</p>
 */
@DisplayName("DefaultGreetingService")
class DefaultGreetingServiceTest {

    private GreetingProperties properties;
    private DefaultGreetingService service;

    @BeforeEach
    void setUp() {
        properties = new GreetingProperties();
        service = new DefaultGreetingService(properties);
    }

    @Nested
    @DisplayName("Constructor")
    class ConstructorTests {

        @Test
        @DisplayName("sollte Properties akzeptieren")
        void shouldAcceptProperties() {
            DefaultGreetingService svc = new DefaultGreetingService(properties);
            assertThat(svc.getProperties()).isSameAs(properties);
        }

        @Test
        @DisplayName("sollte Exception werfen bei null Properties")
        void shouldThrowExceptionForNullProperties() {
            assertThatThrownBy(() -> new DefaultGreetingService(null))
                .isInstanceOf(IllegalArgumentException.class)
                .hasMessageContaining("must not be null");
        }
    }

    @Nested
    @DisplayName("greet() mit Default-Properties")
    class GreetWithDefaults {

        @Test
        @DisplayName("sollte Standard-Begrüßung zurückgeben")
        void shouldReturnDefaultGreeting() {
            String result = service.greet("Franz");
            assertThat(result).isEqualTo("Hello, Franz!");
        }

        @Test
        @DisplayName("sollte verschiedene Namen verarbeiten")
        void shouldHandleDifferentNames() {
            assertThat(service.greet("Nova")).isEqualTo("Hello, Nova!");
            assertThat(service.greet("Elyndra")).isEqualTo("Hello, Elyndra!");
            assertThat(service.greet("Code Sentinel")).isEqualTo("Hello, Code Sentinel!");
        }

        @Test
        @DisplayName("sollte Whitespace im Namen trimmen")
        void shouldTrimWhitespaceInName() {
            assertThat(service.greet("  Franz  ")).isEqualTo("Hello, Franz!");
        }
    }

    @Nested
    @DisplayName("greet() mit custom Message")
    class GreetWithCustomMessage {

        @Test
        @DisplayName("sollte custom Message verwenden")
        void shouldUseCustomMessage() {
            properties.setMessage("Guten Tag");
            String result = service.greet("Franz");
            assertThat(result).isEqualTo("Guten Tag, Franz!");
        }

        @Test
        @DisplayName("sollte verschiedene Messages unterstützen")
        void shouldSupportDifferentMessages() {
            properties.setMessage("Welcome");
            assertThat(service.greet("Nova")).isEqualTo("Welcome, Nova!");

            properties.setMessage("Hallo");
            assertThat(service.greet("Nova")).isEqualTo("Hallo, Nova!");
        }
    }

    @Nested
    @DisplayName("greet() mit Prefix")
    class GreetWithPrefix {

        @Test
        @DisplayName("sollte Prefix voranstellen")
        void shouldPrependPrefix() {
            properties.setPrefix("[VIP]");
            String result = service.greet("Franz");
            assertThat(result).isEqualTo("[VIP] Hello, Franz!");
        }

        @Test
        @DisplayName("sollte verschiedene Prefixes unterstützen")
        void shouldSupportDifferentPrefixes() {
            properties.setPrefix("[PREMIUM]");
            assertThat(service.greet("Nova")).isEqualTo("[PREMIUM] Hello, Nova!");

            properties.setPrefix("***");
            assertThat(service.greet("Nova")).isEqualTo("*** Hello, Nova!");
        }

        @Test
        @DisplayName("sollte leeren Prefix ignorieren")
        void shouldIgnoreEmptyPrefix() {
            properties.setPrefix("");
            assertThat(service.greet("Franz")).isEqualTo("Hello, Franz!");
        }

        @Test
        @DisplayName("sollte null Prefix ignorieren")
        void shouldIgnoreNullPrefix() {
            properties.setPrefix(null);
            assertThat(service.greet("Franz")).isEqualTo("Hello, Franz!");
        }
    }

    @Nested
    @DisplayName("greet() mit Uppercase")
    class GreetWithUppercase {

        @Test
        @DisplayName("sollte Ausgabe in Großbuchstaben wandeln")
        void shouldConvertToUppercase() {
            properties.setUppercase(true);
            String result = service.greet("Franz");
            assertThat(result).isEqualTo("HELLO, FRANZ!");
        }

        @Test
        @DisplayName("sollte mit Prefix in Großbuchstaben funktionieren")
        void shouldWorkWithPrefixUppercase() {
            properties.setPrefix("[VIP]");
            properties.setUppercase(true);
            assertThat(service.greet("Franz")).isEqualTo("[VIP] HELLO, FRANZ!");
        }
    }

    @Nested
    @DisplayName("greet() mit Format-Optionen")
    class GreetWithFormat {

        @Test
        @DisplayName("sollte custom Separator verwenden")
        void shouldUseCustomSeparator() {
            properties.getFormat().setSeparator(" -> ");
            String result = service.greet("Franz");
            assertThat(result).isEqualTo("Hello -> Franz!");
        }

        @Test
        @DisplayName("sollte custom Suffix verwenden")
        void shouldUseCustomSuffix() {
            properties.getFormat().setSuffix("?");
            String result = service.greet("Franz");
            assertThat(result).isEqualTo("Hello, Franz?");
        }

        @Test
        @DisplayName("sollte Emoji hinzufügen")
        void shouldAddEmoji() {
            properties.getFormat().setEmoji("👋");
            String result = service.greet("Franz");
            assertThat(result).isEqualTo("👋 Hello, Franz! 👋");
        }

        @Test
        @DisplayName("sollte leeres Emoji ignorieren")
        void shouldIgnoreEmptyEmoji() {
            properties.getFormat().setEmoji("");
            assertThat(service.greet("Franz")).isEqualTo("Hello, Franz!");
        }

        @Test
        @DisplayName("sollte null Emoji ignorieren")
        void shouldIgnoreNullEmoji() {
            properties.getFormat().setEmoji(null);
            assertThat(service.greet("Franz")).isEqualTo("Hello, Franz!");
        }
    }

    @Nested
    @DisplayName("greet() mit allen Optionen kombiniert")
    class GreetWithAllOptions {

        @Test
        @DisplayName("sollte alle Optionen korrekt kombinieren")
        void shouldCombineAllOptions() {
            properties.setMessage("Willkommen");
            properties.setPrefix("[PREMIUM]");
            properties.setUppercase(true);
            properties.getFormat().setEmoji("🎉");
            properties.getFormat().setSeparator(" - ");
            properties.getFormat().setSuffix("!!!");

            String result = service.greet("Franz");
            
            // Erwartung: Emoji + Prefix + Message + Separator + Name + Suffix (uppercase vor emoji)
            assertThat(result).isEqualTo("🎉 [PREMIUM] WILLKOMMEN - FRANZ!!! 🎉");
        }

        @Test
        @DisplayName("sollte korrekte Reihenfolge der Transformationen einhalten")
        void shouldMaintainCorrectTransformationOrder() {
            // Uppercase wird VOR Emoji angewendet, damit Emoji nicht betroffen ist
            properties.setUppercase(true);
            properties.getFormat().setEmoji("👋");

            String result = service.greet("Franz");
            
            // Emoji sollte nicht uppercase sein!
            assertThat(result).isEqualTo("👋 HELLO, FRANZ! 👋");
        }
    }

    @Nested
    @DisplayName("greet() Edge Cases")
    class GreetEdgeCases {

        @ParameterizedTest
        @NullAndEmptySource
        @ValueSource(strings = {"   ", "\t", "\n"})
        @DisplayName("sollte Exception werfen für ungültige Namen")
        void shouldThrowExceptionForInvalidNames(String invalidName) {
            assertThatThrownBy(() -> service.greet(invalidName))
                .isInstanceOf(IllegalArgumentException.class)
                .hasMessageContaining("Name must not be null or empty");
        }

        @Test
        @DisplayName("sollte Namen mit Sonderzeichen verarbeiten")
        void shouldHandleSpecialCharactersInName() {
            assertThat(service.greet("Franz-Martin")).isEqualTo("Hello, Franz-Martin!");
            assertThat(service.greet("O'Brien")).isEqualTo("Hello, O'Brien!");
            assertThat(service.greet("名前")).isEqualTo("Hello, 名前!");
        }

        @Test
        @DisplayName("sollte sehr lange Namen verarbeiten")
        void shouldHandleLongNames() {
            String longName = "A".repeat(1000);
            String result = service.greet(longName);
            assertThat(result).startsWith("Hello, ");
            assertThat(result).endsWith("!");
            assertThat(result).contains(longName);
        }

        @Test
        @DisplayName("sollte einzeichige Namen verarbeiten")
        void shouldHandleSingleCharacterNames() {
            assertThat(service.greet("X")).isEqualTo("Hello, X!");
        }
    }

    @Nested
    @DisplayName("getProperties()")
    class GetPropertiesTests {

        @Test
        @DisplayName("sollte injizierte Properties zurückgeben")
        void shouldReturnInjectedProperties() {
            assertThat(service.getProperties()).isSameAs(properties);
        }
    }
}
