package com.javafleet.greeting;

import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Nested;
import org.junit.jupiter.api.Test;
import org.springframework.boot.autoconfigure.AutoConfigurations;
import org.springframework.boot.test.context.runner.ApplicationContextRunner;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import static org.assertj.core.api.Assertions.assertThat;

/**
 * Integration Tests für {@link GreetingAutoConfiguration}.
 * 
 * <p>Verwendet {@link ApplicationContextRunner} – den empfohlenen Weg
 * um Spring Boot Auto-Configuration zu testen.</p>
 * 
 * <h2>Warum ApplicationContextRunner?</h2>
 * <ul>
 *   <li>Schnell: Erstellt keinen echten Application Context</li>
 *   <li>Isoliert: Jeder Test startet mit frischem Context</li>
 *   <li>Flexibel: Properties können pro Test gesetzt werden</li>
 *   <li>Fluent API: Lesbare, chainbare Assertions</li>
 * </ul>
 * 
 * @see ApplicationContextRunner
 */
@DisplayName("GreetingAutoConfiguration")
class GreetingAutoConfigurationTest {

    /**
     * ApplicationContextRunner mit unserer Auto-Configuration.
     * Wird als Basis für alle Tests verwendet.
     */
    private final ApplicationContextRunner contextRunner = new ApplicationContextRunner()
            .withConfiguration(AutoConfigurations.of(GreetingAutoConfiguration.class));

    @Nested
    @DisplayName("Bean-Erstellung")
    class BeanCreation {

        @Test
        @DisplayName("sollte GreetingService Bean erstellen")
        void shouldCreateGreetingServiceBean() {
            contextRunner.run(context -> {
                assertThat(context).hasSingleBean(GreetingService.class);
                assertThat(context).hasSingleBean(DefaultGreetingService.class);
            });
        }

        @Test
        @DisplayName("sollte GreetingProperties Bean erstellen")
        void shouldCreateGreetingPropertiesBean() {
            contextRunner.run(context -> {
                assertThat(context).hasSingleBean(GreetingProperties.class);
            });
        }

        @Test
        @DisplayName("GreetingService sollte funktionieren")
        void greetingServiceShouldWork() {
            contextRunner.run(context -> {
                GreetingService service = context.getBean(GreetingService.class);
                assertThat(service.greet("Test")).isEqualTo("Hello, Test!");
            });
        }
    }

    @Nested
    @DisplayName("Property Binding")
    class PropertyBinding {

        @Test
        @DisplayName("sollte greeting.message binden")
        void shouldBindMessage() {
            contextRunner
                .withPropertyValues("greeting.message=Guten Tag")
                .run(context -> {
                    GreetingService service = context.getBean(GreetingService.class);
                    assertThat(service.greet("Franz")).isEqualTo("Guten Tag, Franz!");
                });
        }

        @Test
        @DisplayName("sollte greeting.prefix binden")
        void shouldBindPrefix() {
            contextRunner
                .withPropertyValues("greeting.prefix=[VIP]")
                .run(context -> {
                    GreetingService service = context.getBean(GreetingService.class);
                    assertThat(service.greet("Franz")).isEqualTo("[VIP] Hello, Franz!");
                });
        }

        @Test
        @DisplayName("sollte greeting.uppercase binden")
        void shouldBindUppercase() {
            contextRunner
                .withPropertyValues("greeting.uppercase=true")
                .run(context -> {
                    GreetingService service = context.getBean(GreetingService.class);
                    assertThat(service.greet("Franz")).isEqualTo("HELLO, FRANZ!");
                });
        }

        @Test
        @DisplayName("sollte nested greeting.format.* Properties binden")
        void shouldBindNestedFormatProperties() {
            contextRunner
                .withPropertyValues(
                    "greeting.format.emoji=👋",
                    "greeting.format.separator= -> ",
                    "greeting.format.suffix=???"
                )
                .run(context -> {
                    GreetingService service = context.getBean(GreetingService.class);
                    assertThat(service.greet("Franz")).isEqualTo("👋 Hello -> Franz??? 👋");
                });
        }

        @Test
        @DisplayName("sollte alle Properties kombinieren")
        void shouldCombineAllProperties() {
            contextRunner
                .withPropertyValues(
                    "greeting.message=Welcome",
                    "greeting.prefix=[PREMIUM]",
                    "greeting.uppercase=true",
                    "greeting.format.emoji=🎉"
                )
                .run(context -> {
                    GreetingService service = context.getBean(GreetingService.class);
                    assertThat(service.greet("Franz")).isEqualTo("🎉 [PREMIUM] WELCOME, FRANZ! 🎉");
                });
        }
    }

    @Nested
    @DisplayName("@ConditionalOnProperty (greeting.enabled)")
    class ConditionalOnProperty {

        @Test
        @DisplayName("sollte Bean erstellen wenn enabled nicht gesetzt ist (matchIfMissing=true)")
        void shouldCreateBeanWhenEnabledNotSet() {
            contextRunner.run(context -> {
                assertThat(context).hasSingleBean(GreetingService.class);
            });
        }

        @Test
        @DisplayName("sollte Bean erstellen wenn enabled=true")
        void shouldCreateBeanWhenEnabledTrue() {
            contextRunner
                .withPropertyValues("greeting.enabled=true")
                .run(context -> {
                    assertThat(context).hasSingleBean(GreetingService.class);
                });
        }

        @Test
        @DisplayName("sollte KEINEN Bean erstellen wenn enabled=false")
        void shouldNotCreateBeanWhenEnabledFalse() {
            contextRunner
                .withPropertyValues("greeting.enabled=false")
                .run(context -> {
                    assertThat(context).doesNotHaveBean(GreetingService.class);
                    assertThat(context).doesNotHaveBean(DefaultGreetingService.class);
                });
        }
    }

    @Nested
    @DisplayName("@ConditionalOnMissingBean (Überschreibbarkeit)")
    class ConditionalOnMissingBean {

        @Test
        @DisplayName("sollte Custom-Implementierung bevorzugen")
        void shouldPreferCustomImplementation() {
            contextRunner
                .withUserConfiguration(CustomGreetingServiceConfig.class)
                .run(context -> {
                    assertThat(context).hasSingleBean(GreetingService.class);
                    assertThat(context).doesNotHaveBean(DefaultGreetingService.class);
                    
                    GreetingService service = context.getBean(GreetingService.class);
                    assertThat(service.greet("Test")).isEqualTo("CUSTOM: Test");
                });
        }

        @Test
        @DisplayName("Custom-Implementierung sollte Properties nicht beeinflussen")
        void customImplementationShouldNotAffectProperties() {
            contextRunner
                .withUserConfiguration(CustomGreetingServiceConfig.class)
                .withPropertyValues("greeting.message=Ignored")
                .run(context -> {
                    // Properties Bean sollte trotzdem existieren
                    assertThat(context).hasSingleBean(GreetingProperties.class);
                    
                    // Aber Custom-Service verwendet sie nicht
                    GreetingService service = context.getBean(GreetingService.class);
                    assertThat(service.greet("Test")).isEqualTo("CUSTOM: Test");
                });
        }

        /**
         * Test-Configuration mit Custom GreetingService.
         */
        @Configuration(proxyBeanMethods = false)
        static class CustomGreetingServiceConfig {
            
            @Bean
            GreetingService greetingService() {
                return name -> "CUSTOM: " + name;
            }
        }
    }

    @Nested
    @DisplayName("Edge Cases")
    class EdgeCases {

        @Test
        @DisplayName("sollte mit leeren Properties funktionieren")
        void shouldWorkWithEmptyProperties() {
            contextRunner
                .withPropertyValues(
                    "greeting.message=",
                    "greeting.prefix=",
                    "greeting.format.emoji="
                )
                .run(context -> {
                    GreetingService service = context.getBean(GreetingService.class);
                    // Leere Message ergibt: ", Name!"
                    assertThat(service.greet("Franz")).isEqualTo(", Franz!");
                });
        }

        @Test
        @DisplayName("sollte relaxed binding unterstützen")
        void shouldSupportRelaxedBinding() {
            // Spring Boot's Relaxed Binding erlaubt verschiedene Schreibweisen
            contextRunner
                .withPropertyValues("GREETING_MESSAGE=ENV_STYLE")
                .run(context -> {
                    GreetingProperties props = context.getBean(GreetingProperties.class);
                    assertThat(props.getMessage()).isEqualTo("ENV_STYLE");
                });
        }
    }

    @Nested
    @DisplayName("Context Lifecycle")
    class ContextLifecycle {

        @Test
        @DisplayName("sollte Context sauber starten")
        void shouldStartContextCleanly() {
            contextRunner.run(context -> {
                assertThat(context).hasNotFailed();
            });
        }

        @Test
        @DisplayName("sollte keine Startup-Exceptions werfen")
        void shouldNotThrowStartupExceptions() {
            contextRunner.run(context -> {
                assertThat(context.getStartupFailure()).isNull();
            });
        }
    }
}
