package de.javafleet.gui;

import com.formdev.flatlaf.FlatDarkLaf;
import javax.swing.*;
import java.awt.*;

/**
 * HalloSwing - Dein erstes Swing-Programm!
 * 
 * Tag 1 des Kurses "Java Anwendungsentwicklung"
 * Java Fleet Systems Consulting - java-developer.online
 * 
 * Was du hier lernst:
 * - Ein Fenster mit JFrame erstellen
 * - Komponenten hinzufügen (JLabel, JButton)
 * - Event-Handling mit ActionListener
 * - Look-and-Feel setzen
 * 
 * @author Franz-Martin (CTO, Java Fleet Systems Consulting)
 */
public class HalloSwing {
    
    public static void main(String[] args) {
        // ══════════════════════════════════════════════════════════════
        // WICHTIG: Swing-Code IMMER im Event Dispatch Thread (EDT)!
        // 
        // Warum? Swing ist nicht thread-safe. Wenn du GUI-Code außerhalb
        // des EDT ausführst, bekommst du Race Conditions - manchmal.
        // Das "manchmal" macht das Debugging zur Hölle.
        // ══════════════════════════════════════════════════════════════
        SwingUtilities.invokeLater(() -> {
            // Look-and-Feel setzen (VOR dem Erstellen von Komponenten!)
            setzeLookAndFeel();
            
            // Fenster erstellen und anzeigen
            erstelleUndZeigeFenster();
        });
    }
    
    /**
     * Setzt das Look-and-Feel für die Anwendung.
     * 
     * Optionen:
     * - UIManager.getSystemLookAndFeelClassName() → Sieht aus wie das OS
     * - "javax.swing.plaf.nimbus.NimbusLookAndFeel" → Moderner Java-Look
     * - FlatDarkLaf → Modernes Dark Theme (benötigt FlatLaf Dependency)
     */
    private static void setzeLookAndFeel() {
        try {
            // FlatLaf Dark Theme - sieht VIEL moderner aus als Standard-Swing!
            UIManager.setLookAndFeel(new FlatDarkLaf());
            
            // Alternative: System Look-and-Feel (sieht aus wie Windows/Mac)
            // UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
            
            // Alternative: Nimbus (Java's "moderner" Look von 2011...)
            // UIManager.setLookAndFeel("javax.swing.plaf.nimbus.NimbusLookAndFeel");
            
        } catch (Exception e) {
            // Wenn das L&F nicht gesetzt werden kann, nimm den Standard
            System.err.println("Look-and-Feel konnte nicht gesetzt werden: " + e.getMessage());
        }
    }
    
    /**
     * Erstellt das Hauptfenster mit allen Komponenten.
     */
    private static void erstelleUndZeigeFenster() {
        // ══════════════════════════════════════════════════════════════
        // 1. FENSTER ERSTELLEN
        // ══════════════════════════════════════════════════════════════
        JFrame fenster = new JFrame("🚀 Willkommen bei Java Anwendungsentwicklung!");
        
        // Was passiert, wenn der User das Fenster schließt?
        // EXIT_ON_CLOSE → Anwendung beendet
        // DISPOSE_ON_CLOSE → Nur dieses Fenster schließen
        // DO_NOTHING_ON_CLOSE → Nichts tun (für eigene Bestätigungs-Dialoge)
        fenster.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        
        // ══════════════════════════════════════════════════════════════
        // 2. HAUPTPANEL MIT LAYOUT
        // ══════════════════════════════════════════════════════════════
        JPanel hauptPanel = new JPanel();
        hauptPanel.setLayout(new BorderLayout(10, 10)); // 10px Abstand
        hauptPanel.setBorder(BorderFactory.createEmptyBorder(20, 20, 20, 20));
        
        // ══════════════════════════════════════════════════════════════
        // 3. KOMPONENTEN ERSTELLEN
        // ══════════════════════════════════════════════════════════════
        
        // Titel-Label
        JLabel titelLabel = new JLabel(
            "<html><center>" +
            "<h1>🖥️ Hallo Swing!</h1>" +
            "<p>Dein erstes Desktop-Programm mit Java</p>" +
            "</center></html>",
            SwingConstants.CENTER
        );
        
        // Info-Text
        JTextArea infoText = new JTextArea();
        infoText.setText(
            """
            ╔══════════════════════════════════════════════════════════════╗
            ║  Was du hier siehst:                                         ║
            ║                                                              ║
            ║  • Ein JFrame (das Fenster)                                  ║
            ║  • Ein JPanel (Container für andere Komponenten)             ║
            ║  • Ein JLabel (der Titel oben)                               ║
            ║  • Eine JTextArea (dieser Text hier)                         ║
            ║  • Zwei JButtons (unten)                                     ║
            ║                                                              ║
            ║  Das Look-and-Feel ist "FlatLaf Dark" - viel moderner        ║
            ║  als das Standard-Swing von 1998!                            ║
            ╚══════════════════════════════════════════════════════════════╝
            
            Franz-Martins Frage an dich:
            
            "Dieses Fenster läuft lokal auf deinem Rechner. 
            Es braucht kein Internet, keinen Server.
            
            Wann ist das ein Vorteil? Wann ein Nachteil?"
            
            Denk darüber nach, bevor du auf 'Weiter' klickst!
            """
        );
        infoText.setEditable(false);
        infoText.setFont(new Font(Font.MONOSPACED, Font.PLAIN, 12));
        
        // Scroll-Pane für den Text (falls er zu lang wird)
        JScrollPane scrollPane = new JScrollPane(infoText);
        
        // Button-Panel
        JPanel buttonPanel = new JPanel(new FlowLayout(FlowLayout.CENTER, 20, 10));
        
        JButton btnInfo = new JButton("ℹ️ Mehr erfahren");
        JButton btnWeiter = new JButton("▶️ Weiter zu Tag 2");
        
        // ══════════════════════════════════════════════════════════════
        // 4. EVENT-HANDLING (Das Observer-Pattern!)
        // ══════════════════════════════════════════════════════════════
        
        // Lambda-Syntax (modern, empfohlen)
        btnInfo.addActionListener(e -> {
            JOptionPane.showMessageDialog(
                fenster,
                """
                ╔══════════════════════════════════════════════════════════╗
                ║  Warum lernst du Swing?                                  ║
                ╠══════════════════════════════════════════════════════════╣
                ║                                                          ║
                ║  1. Du wirst Legacy-Systeme warten müssen                ║
                ║     (Banken, Versicherungen, Behörden)                   ║
                ║                                                          ║
                ║  2. Die Konzepte (Event-Handling, MVC) sind überall      ║
                ║     gleich - auch in React, Vue, Angular!                ║
                ║                                                          ║
                ║  3. Du verstehst, WARUM das Web gewonnen hat             ║
                ║                                                          ║
                ║  "Legacy ist nicht schlecht.                             ║
                ║   Legacy bedeutet: Es hat funktioniert."                 ║
                ║                           — Franz-Martin                 ║
                ╚══════════════════════════════════════════════════════════╝
                """,
                "Warum Desktop-Entwicklung lernen?",
                JOptionPane.INFORMATION_MESSAGE
            );
        });
        
        btnWeiter.addActionListener(e -> {
            int antwort = JOptionPane.showConfirmDialog(
                fenster,
                "Tag 1 ist ein Theorie-Tag.\n\n" +
                "Ab Tag 2 wird's praktisch - du baust dann\n" +
                "echte GUI-Anwendungen!\n\n" +
                "Bereit für morgen?",
                "Weiter zu Tag 2?",
                JOptionPane.YES_NO_OPTION,
                JOptionPane.QUESTION_MESSAGE
            );
            
            if (antwort == JOptionPane.YES_OPTION) {
                JOptionPane.showMessageDialog(
                    fenster,
                    "Super! 🎉\n\n" +
                    "Lies den Blogpost zu Tag 1 zu Ende,\n" +
                    "dann bist du bereit für Tag 2!\n\n" +
                    "Bis morgen!\n" +
                    "— Franz-Martin",
                    "Bis morgen!",
                    JOptionPane.INFORMATION_MESSAGE
                );
            }
        });
        
        // ══════════════════════════════════════════════════════════════
        // 5. KOMPONENTEN ZUSAMMENBAUEN
        // ══════════════════════════════════════════════════════════════
        buttonPanel.add(btnInfo);
        buttonPanel.add(btnWeiter);
        
        hauptPanel.add(titelLabel, BorderLayout.NORTH);
        hauptPanel.add(scrollPane, BorderLayout.CENTER);
        hauptPanel.add(buttonPanel, BorderLayout.SOUTH);
        
        fenster.add(hauptPanel);
        
        // ══════════════════════════════════════════════════════════════
        // 6. FENSTER ANZEIGEN
        // ══════════════════════════════════════════════════════════════
        fenster.setSize(700, 550);
        fenster.setLocationRelativeTo(null); // Fenster in Bildschirmmitte
        fenster.setVisible(true);
    }
}
