package de.javafleet.gui;

import com.formdev.flatlaf.FlatDarkLaf;
import javax.swing.*;
import javax.swing.border.TitledBorder;
import java.awt.*;

/**
 * KomponentenDemo - Alle wichtigen Swing-Komponenten auf einen Blick.
 * 
 * Tag 2 des Kurses "Java Anwendungsentwicklung"
 * Java Fleet Systems Consulting - java-developer.online
 * 
 * Diese Demo zeigt dir:
 * - JLabel, JButton, JTextField, JPasswordField
 * - JTextArea mit JScrollPane
 * - JCheckBox, JRadioButton mit ButtonGroup
 * - JComboBox, JList
 * - JSlider, JProgressBar
 * - JSpinner
 * 
 * @author Franz-Martin (CTO, Java Fleet Systems Consulting)
 */
public class KomponentenDemo {
    
    public static void main(String[] args) {
        SwingUtilities.invokeLater(() -> {
            try {
                UIManager.setLookAndFeel(new FlatDarkLaf());
            } catch (Exception e) {
                System.err.println("FlatLaf konnte nicht geladen werden.");
            }
            
            new KomponentenDemo().erstelleDemo();
        });
    }
    
    private void erstelleDemo() {
        JFrame frame = new JFrame("Swing Komponenten-Demo | Tag 2");
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        
        // Hauptpanel mit Tabs für bessere Übersicht
        JTabbedPane tabbedPane = new JTabbedPane();
        
        // Tab 1: Text-Komponenten
        tabbedPane.addTab("📝 Text", erstelleTextPanel());
        
        // Tab 2: Buttons & Auswahl
        tabbedPane.addTab("🔘 Auswahl", erstelleAuswahlPanel());
        
        // Tab 3: Listen & Dropdowns
        tabbedPane.addTab("📋 Listen", erstelleListenPanel());
        
        // Tab 4: Slider & Progress
        tabbedPane.addTab("📊 Slider", erstelleSliderPanel());
        
        frame.add(tabbedPane);
        frame.setSize(700, 500);
        frame.setLocationRelativeTo(null);
        frame.setVisible(true);
    }
    
    // ══════════════════════════════════════════════════════════════
    // TAB 1: TEXT-KOMPONENTEN
    // ══════════════════════════════════════════════════════════════
    
    private JPanel erstelleTextPanel() {
        JPanel panel = new JPanel();
        panel.setLayout(new BoxLayout(panel, BoxLayout.Y_AXIS));
        panel.setBorder(BorderFactory.createEmptyBorder(15, 15, 15, 15));
        
        // ──────────────────────────────────────────────────────────
        // JLabel - Nur zur Anzeige, nicht editierbar
        // ──────────────────────────────────────────────────────────
        JPanel labelPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        labelPanel.setBorder(createBorder("JLabel - Text anzeigen"));
        
        JLabel einfachesLabel = new JLabel("Ein einfaches Label");
        JLabel htmlLabel = new JLabel("<html><b>HTML</b> <i>funktioniert</i> <u>auch</u>!</html>");
        JLabel iconLabel = new JLabel("Mit Icon:", JLabel.LEFT);
        // In echter App: iconLabel.setIcon(new ImageIcon("pfad/zum/icon.png"));
        
        labelPanel.add(einfachesLabel);
        labelPanel.add(Box.createHorizontalStrut(20));
        labelPanel.add(htmlLabel);
        labelPanel.add(Box.createHorizontalStrut(20));
        labelPanel.add(iconLabel);
        
        // ──────────────────────────────────────────────────────────
        // JTextField - Einzeilige Texteingabe
        // ──────────────────────────────────────────────────────────
        JPanel textFieldPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        textFieldPanel.setBorder(createBorder("JTextField - Einzeilige Eingabe"));
        
        JTextField normalField = new JTextField(15);
        normalField.setToolTipText("Normales Textfeld");
        
        JTextField mitTextField = new JTextField("Vorausgefüllt", 15);
        
        JTextField readOnlyField = new JTextField("Nicht editierbar", 15);
        readOnlyField.setEditable(false);
        
        textFieldPanel.add(new JLabel("Normal:"));
        textFieldPanel.add(normalField);
        textFieldPanel.add(new JLabel("Vorausgefüllt:"));
        textFieldPanel.add(mitTextField);
        textFieldPanel.add(new JLabel("ReadOnly:"));
        textFieldPanel.add(readOnlyField);
        
        // ──────────────────────────────────────────────────────────
        // JPasswordField - Für Passwörter (zeigt Punkte statt Text)
        // ──────────────────────────────────────────────────────────
        JPanel passwordPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        passwordPanel.setBorder(createBorder("JPasswordField - Passwort-Eingabe"));
        
        JPasswordField passwordField = new JPasswordField(15);
        passwordField.setToolTipText("Zeigt ● statt Buchstaben");
        
        JButton showPassword = new JButton("Zeigen");
        showPassword.addActionListener(e -> {
            // ACHTUNG: getPassword() gibt char[] zurück, nicht String!
            // Das ist Absicht - Strings bleiben im Speicher, char[] kann überschrieben werden.
            char[] password = passwordField.getPassword();
            JOptionPane.showMessageDialog(null, "Passwort: " + new String(password));
            // In echter App: Arrays.fill(password, '0'); // Überschreiben!
        });
        
        passwordPanel.add(new JLabel("Passwort:"));
        passwordPanel.add(passwordField);
        passwordPanel.add(showPassword);
        
        // ──────────────────────────────────────────────────────────
        // JTextArea - Mehrzeilige Texteingabe
        // ──────────────────────────────────────────────────────────
        JPanel textAreaPanel = new JPanel(new BorderLayout());
        textAreaPanel.setBorder(createBorder("JTextArea - Mehrzeilige Eingabe"));
        
        JTextArea textArea = new JTextArea(5, 30);
        textArea.setText("Hier kannst du mehrere Zeilen schreiben...\n\nZeile 3\nZeile 4");
        textArea.setLineWrap(true);        // Automatischer Zeilenumbruch
        textArea.setWrapStyleWord(true);   // Umbruch nur bei Wörtern
        
        // WICHTIG: JTextArea IMMER in JScrollPane packen!
        JScrollPane scrollPane = new JScrollPane(textArea);
        textAreaPanel.add(scrollPane, BorderLayout.CENTER);
        
        // Alles zusammenbauen
        panel.add(labelPanel);
        panel.add(Box.createVerticalStrut(10));
        panel.add(textFieldPanel);
        panel.add(Box.createVerticalStrut(10));
        panel.add(passwordPanel);
        panel.add(Box.createVerticalStrut(10));
        panel.add(textAreaPanel);
        
        return panel;
    }
    
    // ══════════════════════════════════════════════════════════════
    // TAB 2: BUTTONS & AUSWAHL-KOMPONENTEN
    // ══════════════════════════════════════════════════════════════
    
    private JPanel erstelleAuswahlPanel() {
        JPanel panel = new JPanel();
        panel.setLayout(new BoxLayout(panel, BoxLayout.Y_AXIS));
        panel.setBorder(BorderFactory.createEmptyBorder(15, 15, 15, 15));
        
        // ──────────────────────────────────────────────────────────
        // JButton - Der Klassiker
        // ──────────────────────────────────────────────────────────
        JPanel buttonPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        buttonPanel.setBorder(createBorder("JButton - Schaltflächen"));
        
        JButton normalButton = new JButton("Normal");
        normalButton.addActionListener(e -> 
            JOptionPane.showMessageDialog(null, "Button geklickt!"));
        
        JButton disabledButton = new JButton("Deaktiviert");
        disabledButton.setEnabled(false);
        
        JButton htmlButton = new JButton("<html><b>HTML</b><br>Button</html>");
        
        buttonPanel.add(normalButton);
        buttonPanel.add(disabledButton);
        buttonPanel.add(htmlButton);
        
        // ──────────────────────────────────────────────────────────
        // JCheckBox - Mehrfachauswahl
        // ──────────────────────────────────────────────────────────
        JPanel checkBoxPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        checkBoxPanel.setBorder(createBorder("JCheckBox - Mehrfachauswahl"));
        
        JCheckBox check1 = new JCheckBox("Java");
        JCheckBox check2 = new JCheckBox("Python", true); // Vorausgewählt
        JCheckBox check3 = new JCheckBox("JavaScript");
        
        // Status abfragen
        JButton checkStatus = new JButton("Status?");
        checkStatus.addActionListener(e -> {
            String msg = "Ausgewählt:\n";
            if (check1.isSelected()) msg += "- Java\n";
            if (check2.isSelected()) msg += "- Python\n";
            if (check3.isSelected()) msg += "- JavaScript\n";
            JOptionPane.showMessageDialog(null, msg);
        });
        
        checkBoxPanel.add(check1);
        checkBoxPanel.add(check2);
        checkBoxPanel.add(check3);
        checkBoxPanel.add(checkStatus);
        
        // ──────────────────────────────────────────────────────────
        // JRadioButton - Einzelauswahl (mit ButtonGroup!)
        // ──────────────────────────────────────────────────────────
        JPanel radioPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        radioPanel.setBorder(createBorder("JRadioButton - Einzelauswahl"));
        
        JRadioButton radio1 = new JRadioButton("Klein");
        JRadioButton radio2 = new JRadioButton("Mittel", true); // Vorausgewählt
        JRadioButton radio3 = new JRadioButton("Groß");
        
        // WICHTIG: ButtonGroup sorgt dafür, dass nur EINER ausgewählt ist!
        ButtonGroup sizeGroup = new ButtonGroup();
        sizeGroup.add(radio1);
        sizeGroup.add(radio2);
        sizeGroup.add(radio3);
        
        JButton radioStatus = new JButton("Welche Größe?");
        radioStatus.addActionListener(e -> {
            String size = "Keine";
            if (radio1.isSelected()) size = "Klein";
            if (radio2.isSelected()) size = "Mittel";
            if (radio3.isSelected()) size = "Groß";
            JOptionPane.showMessageDialog(null, "Größe: " + size);
        });
        
        radioPanel.add(radio1);
        radioPanel.add(radio2);
        radioPanel.add(radio3);
        radioPanel.add(radioStatus);
        
        // ──────────────────────────────────────────────────────────
        // JToggleButton - Ein/Aus-Schalter
        // ──────────────────────────────────────────────────────────
        JPanel togglePanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        togglePanel.setBorder(createBorder("JToggleButton - Ein/Aus"));
        
        JToggleButton toggleButton = new JToggleButton("Dark Mode: AUS");
        toggleButton.addActionListener(e -> {
            if (toggleButton.isSelected()) {
                toggleButton.setText("Dark Mode: AN");
            } else {
                toggleButton.setText("Dark Mode: AUS");
            }
        });
        
        togglePanel.add(toggleButton);
        
        // Alles zusammenbauen
        panel.add(buttonPanel);
        panel.add(Box.createVerticalStrut(10));
        panel.add(checkBoxPanel);
        panel.add(Box.createVerticalStrut(10));
        panel.add(radioPanel);
        panel.add(Box.createVerticalStrut(10));
        panel.add(togglePanel);
        
        return panel;
    }
    
    // ══════════════════════════════════════════════════════════════
    // TAB 3: LISTEN & DROPDOWNS
    // ══════════════════════════════════════════════════════════════
    
    private JPanel erstelleListenPanel() {
        JPanel panel = new JPanel(new GridLayout(1, 2, 15, 15));
        panel.setBorder(BorderFactory.createEmptyBorder(15, 15, 15, 15));
        
        // ──────────────────────────────────────────────────────────
        // JComboBox - Dropdown-Auswahl
        // ──────────────────────────────────────────────────────────
        JPanel comboPanel = new JPanel(new BorderLayout(10, 10));
        comboPanel.setBorder(createBorder("JComboBox - Dropdown"));
        
        String[] laender = {"Deutschland", "Österreich", "Schweiz", "Frankreich", "Italien"};
        JComboBox<String> comboBox = new JComboBox<>(laender);
        
        JLabel comboResult = new JLabel("Wähle ein Land...");
        comboBox.addActionListener(e -> {
            String selected = (String) comboBox.getSelectedItem();
            comboResult.setText("Ausgewählt: " + selected);
        });
        
        // Editierbare ComboBox
        JComboBox<String> editableCombo = new JComboBox<>(laender);
        editableCombo.setEditable(true); // User kann eigenen Text eingeben!
        
        JPanel comboContent = new JPanel();
        comboContent.setLayout(new BoxLayout(comboContent, BoxLayout.Y_AXIS));
        comboContent.add(new JLabel("Normal:"));
        comboContent.add(comboBox);
        comboContent.add(Box.createVerticalStrut(10));
        comboContent.add(comboResult);
        comboContent.add(Box.createVerticalStrut(20));
        comboContent.add(new JLabel("Editierbar:"));
        comboContent.add(editableCombo);
        
        comboPanel.add(comboContent, BorderLayout.CENTER);
        
        // ──────────────────────────────────────────────────────────
        // JList - Liste mit Mehrfachauswahl
        // ──────────────────────────────────────────────────────────
        JPanel listPanel = new JPanel(new BorderLayout(10, 10));
        listPanel.setBorder(createBorder("JList - Listenauswahl"));
        
        String[] sprachen = {"Java", "Python", "JavaScript", "C++", "Rust", "Go", "Kotlin"};
        JList<String> list = new JList<>(sprachen);
        list.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
        
        JScrollPane listScroll = new JScrollPane(list);
        
        JButton listStatus = new JButton("Auswahl zeigen");
        listStatus.addActionListener(e -> {
            var selected = list.getSelectedValuesList();
            JOptionPane.showMessageDialog(null, "Ausgewählt: " + selected);
        });
        
        listPanel.add(listScroll, BorderLayout.CENTER);
        listPanel.add(listStatus, BorderLayout.SOUTH);
        
        panel.add(comboPanel);
        panel.add(listPanel);
        
        return panel;
    }
    
    // ══════════════════════════════════════════════════════════════
    // TAB 4: SLIDER & PROGRESS
    // ══════════════════════════════════════════════════════════════
    
    private JPanel erstelleSliderPanel() {
        JPanel panel = new JPanel();
        panel.setLayout(new BoxLayout(panel, BoxLayout.Y_AXIS));
        panel.setBorder(BorderFactory.createEmptyBorder(15, 15, 15, 15));
        
        // ──────────────────────────────────────────────────────────
        // JSlider - Werte per Schieberegler
        // ──────────────────────────────────────────────────────────
        JPanel sliderPanel = new JPanel(new BorderLayout(10, 10));
        sliderPanel.setBorder(createBorder("JSlider - Schieberegler"));
        
        JSlider slider = new JSlider(0, 100, 50); // min, max, start
        slider.setMajorTickSpacing(25);
        slider.setMinorTickSpacing(5);
        slider.setPaintTicks(true);
        slider.setPaintLabels(true);
        
        JLabel sliderValue = new JLabel("Wert: 50");
        slider.addChangeListener(e -> {
            sliderValue.setText("Wert: " + slider.getValue());
        });
        
        sliderPanel.add(slider, BorderLayout.CENTER);
        sliderPanel.add(sliderValue, BorderLayout.SOUTH);
        
        // ──────────────────────────────────────────────────────────
        // JProgressBar - Fortschrittsanzeige
        // ──────────────────────────────────────────────────────────
        JPanel progressPanel = new JPanel(new BorderLayout(10, 10));
        progressPanel.setBorder(createBorder("JProgressBar - Fortschritt"));
        
        JProgressBar progressBar = new JProgressBar(0, 100);
        progressBar.setValue(0);
        progressBar.setStringPainted(true); // Zeigt Prozent-Text
        
        JButton startProgress = new JButton("Simulation starten");
        startProgress.addActionListener(e -> {
            startProgress.setEnabled(false);
            
            // Timer simuliert langsamen Fortschritt
            Timer timer = new Timer(100, null);
            timer.addActionListener(evt -> {
                int value = progressBar.getValue();
                if (value < 100) {
                    progressBar.setValue(value + 2);
                } else {
                    timer.stop();
                    startProgress.setEnabled(true);
                    progressBar.setValue(0);
                    JOptionPane.showMessageDialog(null, "Fertig!");
                }
            });
            timer.start();
        });
        
        progressPanel.add(progressBar, BorderLayout.CENTER);
        progressPanel.add(startProgress, BorderLayout.SOUTH);
        
        // ──────────────────────────────────────────────────────────
        // JSpinner - Zahleneingabe mit Pfeilen
        // ──────────────────────────────────────────────────────────
        JPanel spinnerPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        spinnerPanel.setBorder(createBorder("JSpinner - Zahlen/Datum wählen"));
        
        // Zahlen-Spinner
        SpinnerNumberModel numberModel = new SpinnerNumberModel(25, 0, 100, 1);
        JSpinner numberSpinner = new JSpinner(numberModel);
        
        // Datum-Spinner
        SpinnerDateModel dateModel = new SpinnerDateModel();
        JSpinner dateSpinner = new JSpinner(dateModel);
        JSpinner.DateEditor dateEditor = new JSpinner.DateEditor(dateSpinner, "dd.MM.yyyy");
        dateSpinner.setEditor(dateEditor);
        
        spinnerPanel.add(new JLabel("Zahl:"));
        spinnerPanel.add(numberSpinner);
        spinnerPanel.add(Box.createHorizontalStrut(20));
        spinnerPanel.add(new JLabel("Datum:"));
        spinnerPanel.add(dateSpinner);
        
        // Alles zusammenbauen
        panel.add(sliderPanel);
        panel.add(Box.createVerticalStrut(15));
        panel.add(progressPanel);
        panel.add(Box.createVerticalStrut(15));
        panel.add(spinnerPanel);
        
        return panel;
    }
    
    // ══════════════════════════════════════════════════════════════
    // HILFSMETHODE: Schöner Border mit Titel
    // ══════════════════════════════════════════════════════════════
    
    private TitledBorder createBorder(String title) {
        TitledBorder border = BorderFactory.createTitledBorder(title);
        border.setTitleColor(new Color(102, 187, 106)); // Grün wie in den Grafiken
        return border;
    }
}
