package de.javafleet.gui;

import com.formdev.flatlaf.FlatDarkLaf;
import javax.swing.*;
import java.awt.*;

/**
 * MiniFormular - Ein praktisches Registrierungsformular.
 * 
 * Tag 2 des Kurses "Java Anwendungsentwicklung"
 * Java Fleet Systems Consulting - java-developer.online
 * 
 * Dieses Beispiel zeigt:
 * - Wie man ein echtes Formular aufbaut
 * - Eingaben validieren
 * - Daten verarbeiten
 * - Benutzer-Feedback geben
 * 
 * @author Franz-Martin (CTO, Java Fleet Systems Consulting)
 */
public class MiniFormular extends JFrame {
    
    // ══════════════════════════════════════════════════════════════
    // KOMPONENTEN ALS FELDER
    // ══════════════════════════════════════════════════════════════
    // Warum? Damit wir in verschiedenen Methoden darauf zugreifen können.
    
    private JTextField tfVorname;
    private JTextField tfNachname;
    private JTextField tfEmail;
    private JPasswordField pfPasswort;
    private JPasswordField pfPasswortBestaetigung;
    private JComboBox<String> cbLand;
    private JCheckBox chkNewsletter;
    private JCheckBox chkAgb;
    
    public MiniFormular() {
        super("Registrierung | Java Fleet");
        initComponents();
        setupLayout();
        setupActions();
        setupFrame();
    }
    
    // ══════════════════════════════════════════════════════════════
    // SCHRITT 1: Komponenten initialisieren
    // ══════════════════════════════════════════════════════════════
    
    private void initComponents() {
        // Textfelder
        tfVorname = new JTextField(20);
        tfNachname = new JTextField(20);
        tfEmail = new JTextField(20);
        
        // Passwortfelder
        pfPasswort = new JPasswordField(20);
        pfPasswortBestaetigung = new JPasswordField(20);
        
        // Dropdown
        String[] laender = {
            "-- Bitte wählen --",
            "Deutschland",
            "Österreich", 
            "Schweiz",
            "Andere"
        };
        cbLand = new JComboBox<>(laender);
        
        // Checkboxen
        chkNewsletter = new JCheckBox("Newsletter abonnieren (optional)");
        chkAgb = new JCheckBox("Ich akzeptiere die AGB *");
    }
    
    // ══════════════════════════════════════════════════════════════
    // SCHRITT 2: Layout aufbauen
    // ══════════════════════════════════════════════════════════════
    
    private void setupLayout() {
        // Hauptpanel mit BorderLayout
        JPanel mainPanel = new JPanel(new BorderLayout(10, 10));
        mainPanel.setBorder(BorderFactory.createEmptyBorder(20, 20, 20, 20));
        
        // ──────────────────────────────────────────────────────────
        // OBEN: Titel
        // ──────────────────────────────────────────────────────────
        JLabel titleLabel = new JLabel(
            "<html><h2>🚀 Registrierung</h2>" +
            "<p style='color: gray;'>Erstelle dein Java Fleet Konto</p></html>"
        );
        mainPanel.add(titleLabel, BorderLayout.NORTH);
        
        // ──────────────────────────────────────────────────────────
        // MITTE: Formularfelder mit GridBagLayout
        // ──────────────────────────────────────────────────────────
        JPanel formPanel = new JPanel(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(5, 5, 5, 5);
        gbc.anchor = GridBagConstraints.WEST;
        
        // Zeile 0: Vorname
        gbc.gridx = 0; gbc.gridy = 0;
        formPanel.add(new JLabel("Vorname: *"), gbc);
        gbc.gridx = 1;
        formPanel.add(tfVorname, gbc);
        
        // Zeile 1: Nachname
        gbc.gridx = 0; gbc.gridy = 1;
        formPanel.add(new JLabel("Nachname: *"), gbc);
        gbc.gridx = 1;
        formPanel.add(tfNachname, gbc);
        
        // Zeile 2: E-Mail
        gbc.gridx = 0; gbc.gridy = 2;
        formPanel.add(new JLabel("E-Mail: *"), gbc);
        gbc.gridx = 1;
        formPanel.add(tfEmail, gbc);
        
        // Zeile 3: Passwort
        gbc.gridx = 0; gbc.gridy = 3;
        formPanel.add(new JLabel("Passwort: *"), gbc);
        gbc.gridx = 1;
        formPanel.add(pfPasswort, gbc);
        
        // Zeile 4: Passwort bestätigen
        gbc.gridx = 0; gbc.gridy = 4;
        formPanel.add(new JLabel("Passwort bestätigen: *"), gbc);
        gbc.gridx = 1;
        formPanel.add(pfPasswortBestaetigung, gbc);
        
        // Zeile 5: Land
        gbc.gridx = 0; gbc.gridy = 5;
        formPanel.add(new JLabel("Land: *"), gbc);
        gbc.gridx = 1;
        formPanel.add(cbLand, gbc);
        
        // Zeile 6: Newsletter
        gbc.gridx = 0; gbc.gridy = 6;
        gbc.gridwidth = 2;
        formPanel.add(chkNewsletter, gbc);
        
        // Zeile 7: AGB
        gbc.gridy = 7;
        formPanel.add(chkAgb, gbc);
        
        mainPanel.add(formPanel, BorderLayout.CENTER);
        
        // ──────────────────────────────────────────────────────────
        // UNTEN: Buttons
        // ──────────────────────────────────────────────────────────
        JPanel buttonPanel = new JPanel(new FlowLayout(FlowLayout.RIGHT));
        
        JButton btnAbbrechen = new JButton("Abbrechen");
        btnAbbrechen.addActionListener(e -> System.exit(0));
        
        JButton btnRegistrieren = new JButton("Registrieren");
        btnRegistrieren.setBackground(new Color(66, 165, 245));
        btnRegistrieren.setForeground(Color.WHITE);
        
        buttonPanel.add(btnAbbrechen);
        buttonPanel.add(btnRegistrieren);
        
        // Button-Action in separater Methode (sauberer Code!)
        btnRegistrieren.addActionListener(e -> registrieren());
        
        mainPanel.add(buttonPanel, BorderLayout.SOUTH);
        
        // Zum Frame hinzufügen
        add(mainPanel);
    }
    
    // ══════════════════════════════════════════════════════════════
    // SCHRITT 3: Event-Handling / Actions
    // ══════════════════════════════════════════════════════════════
    
    private void setupActions() {
        // Hier könnten weitere Actions hin, z.B.:
        // - Enter-Taste zum Absenden
        // - Echtzeit-Validierung bei Eingabe
        // - Tooltips
        
        tfVorname.setToolTipText("Dein Vorname");
        tfNachname.setToolTipText("Dein Nachname");
        tfEmail.setToolTipText("Eine gültige E-Mail-Adresse");
        pfPasswort.setToolTipText("Mindestens 8 Zeichen");
    }
    
    // ══════════════════════════════════════════════════════════════
    // SCHRITT 4: Frame konfigurieren
    // ══════════════════════════════════════════════════════════════
    
    private void setupFrame() {
        setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        pack(); // Größe automatisch an Inhalt anpassen
        setLocationRelativeTo(null);
        setResizable(false); // Größe fixieren
    }
    
    // ══════════════════════════════════════════════════════════════
    // REGISTRIERUNG: Validierung und Verarbeitung
    // ══════════════════════════════════════════════════════════════
    
    private void registrieren() {
        // ──────────────────────────────────────────────────────────
        // Schritt 1: Daten sammeln
        // ──────────────────────────────────────────────────────────
        String vorname = tfVorname.getText().trim();
        String nachname = tfNachname.getText().trim();
        String email = tfEmail.getText().trim();
        char[] passwort = pfPasswort.getPassword();
        char[] passwortBestaetigung = pfPasswortBestaetigung.getPassword();
        String land = (String) cbLand.getSelectedItem();
        boolean newsletter = chkNewsletter.isSelected();
        boolean agbAkzeptiert = chkAgb.isSelected();
        
        // ──────────────────────────────────────────────────────────
        // Schritt 2: Validierung
        // ──────────────────────────────────────────────────────────
        StringBuilder fehler = new StringBuilder();
        
        if (vorname.isEmpty()) {
            fehler.append("• Vorname ist erforderlich\n");
        }
        
        if (nachname.isEmpty()) {
            fehler.append("• Nachname ist erforderlich\n");
        }
        
        if (email.isEmpty()) {
            fehler.append("• E-Mail ist erforderlich\n");
        } else if (!email.contains("@") || !email.contains(".")) {
            fehler.append("• E-Mail-Format ungültig\n");
        }
        
        if (passwort.length < 8) {
            fehler.append("• Passwort muss mindestens 8 Zeichen haben\n");
        }
        
        if (!java.util.Arrays.equals(passwort, passwortBestaetigung)) {
            fehler.append("• Passwörter stimmen nicht überein\n");
        }
        
        if ("-- Bitte wählen --".equals(land)) {
            fehler.append("• Bitte wähle ein Land\n");
        }
        
        if (!agbAkzeptiert) {
            fehler.append("• AGB müssen akzeptiert werden\n");
        }
        
        // ──────────────────────────────────────────────────────────
        // Schritt 3: Ergebnis anzeigen
        // ──────────────────────────────────────────────────────────
        if (fehler.length() > 0) {
            // Fehler anzeigen
            JOptionPane.showMessageDialog(
                this,
                "Bitte korrigiere folgende Fehler:\n\n" + fehler,
                "Validierungsfehler",
                JOptionPane.ERROR_MESSAGE
            );
        } else {
            // Erfolg!
            String nachricht = String.format(
                "Registrierung erfolgreich! 🎉\n\n" +
                "Name: %s %s\n" +
                "E-Mail: %s\n" +
                "Land: %s\n" +
                "Newsletter: %s\n\n" +
                "(In einer echten App würde das jetzt\n" +
                "an den Server gesendet werden)",
                vorname, nachname, email, land,
                newsletter ? "Ja" : "Nein"
            );
            
            JOptionPane.showMessageDialog(
                this,
                nachricht,
                "Willkommen bei Java Fleet!",
                JOptionPane.INFORMATION_MESSAGE
            );
            
            // Formular leeren
            clearForm();
        }
        
        // SICHERHEIT: Passwort-Arrays überschreiben!
        java.util.Arrays.fill(passwort, '0');
        java.util.Arrays.fill(passwortBestaetigung, '0');
    }
    
    private void clearForm() {
        tfVorname.setText("");
        tfNachname.setText("");
        tfEmail.setText("");
        pfPasswort.setText("");
        pfPasswortBestaetigung.setText("");
        cbLand.setSelectedIndex(0);
        chkNewsletter.setSelected(false);
        chkAgb.setSelected(false);
        tfVorname.requestFocus();
    }
    
    // ══════════════════════════════════════════════════════════════
    // MAIN-METHODE
    // ══════════════════════════════════════════════════════════════
    
    public static void main(String[] args) {
        SwingUtilities.invokeLater(() -> {
            try {
                UIManager.setLookAndFeel(new FlatDarkLaf());
            } catch (Exception e) {
                System.err.println("FlatLaf konnte nicht geladen werden.");
            }
            
            new MiniFormular().setVisible(true);
        });
    }
}
