# 🚀 Spring Boot Basic - Tag 6

**DI & AOP - Dependency Injection & Aspektorientierte Programmierung**

Von **Elyndra Valen**, Senior Entwicklerin bei Java Fleet Systems Consulting

---

## 📋 Überblick

Dieses Projekt demonstriert die fortgeschrittenen Konzepte von Spring Boot:

- ✅ **Dependency Injection** komplett verstehen
- ✅ **Constructor Injection** (Best Practice mit Lombok)
- ✅ **@Qualifier & @Primary** für multiple Implementierungen
- ✅ **Bean Lifecycle** (@PostConstruct, @PreDestroy)
- ✅ **AOP (Aspect-Oriented Programming)** für Cross-Cutting Concerns
- ✅ **Custom Annotations** (@Timed) erstellen
- ✅ **Automatisches Logging** mit @Aspect
- ✅ **Performance-Messung** mit @Around

---

## 🎯 Was du lernst

### 1. Dependency Injection

**Warum DI?**
- Loose Coupling (lose Kopplung)
- Testbarkeit (Mock-Objekte einfach injizierbar)
- Flexibilität (Implementierung austauschbar)
- Singleton-Management durch Spring

**Die 3 Arten:**
1. **Constructor Injection** ⭐ (BEST PRACTICE!)
   - Immutable (final möglich)
   - Testbar
   - Null-safe
   
2. **Field Injection** ❌ (NICHT EMPFOHLEN!)
   - Nicht testbar
   - Versteckte Dependencies
   
3. **Setter Injection** ⚠️ (selten)
   - Nur für optionale Dependencies

**Beispiel:**
```java
@Controller
@RequiredArgsConstructor  // Lombok generiert Constructor!
public class PersonViewController {
    private final PersonService personService;  // Constructor Injection
}
```

### 2. Multiple Implementierungen

**Problem:** Mehrere Beans vom gleichen Typ

**Lösung 1: @Primary**
```java
@Service
@Primary  // Diese Bean ist Standard!
public class InMemoryPersonService implements PersonService {
    // ...
}
```

**Lösung 2: @Qualifier**
```java
@Qualifier("cachedPersonService")
private final PersonService personService;
```

### 3. Bean Lifecycle

**@PostConstruct** - Nach Dependency Injection
```java
@PostConstruct
public void init() {
    log.info("Bean wird initialisiert...");
    // Setup-Logic: Cache laden, Verbindungen aufbauen
}
```

**@PreDestroy** - Vor Shutdown
```java
@PreDestroy
public void cleanup() {
    log.info("Bean wird heruntergefahren...");
    // Cleanup: Verbindungen schließen, Daten speichern
}
```

### 4. AOP (Aspect-Oriented Programming)

**Was ist AOP?**
AOP löst Cross-Cutting Concerns (Logging, Performance, Security) elegant, ohne Code-Duplikation.

**Advice-Typen:**
- `@Before` - VOR der Methode
- `@After` - NACH der Methode (immer)
- `@AfterReturning` - NACH erfolgreicher Methode
- `@AfterThrowing` - NACH Exception
- `@Around` - VOR UND NACH (am mächtigsten!)

**Logging Aspect:**
```java
@Aspect
@Component
public class LoggingAspect {
    @Before("execution(* com.javafleet.springboot.service.*.*(..))")
    public void logBefore(JoinPoint joinPoint) {
        log.info("→ Calling {}.{}", className, methodName);
    }
}
```

**Performance Aspect mit Custom Annotation:**
```java
@Around("@annotation(com.javafleet.springboot.annotation.Timed)")
public Object measurePerformance(ProceedingJoinPoint joinPoint) {
    long start = System.currentTimeMillis();
    Object result = joinPoint.proceed();
    long duration = System.currentTimeMillis() - start;
    log.info("⏱️ {} took {}ms", methodName, duration);
    return result;
}
```

---

## 🚀 Schnellstart

### Voraussetzungen
- JDK 17 oder höher
- Maven 3.6+
- NetBeans/IntelliJ IDEA/Eclipse

### Installation

1. **Projekt klonen/herunterladen**
   ```bash
   cd spring-boot-basic-tag6
   ```

2. **Dependencies installieren**
   ```bash
   mvn clean install
   ```

3. **Projekt starten**
   ```bash
   mvn spring-boot:run
   ```

4. **Browser öffnen**
   ```
   http://localhost:8080
   ```

---

## 📁 Projektstruktur

```
spring-boot-basic-tag6/
├── src/main/java/com/javafleet/springboot/
│   ├── SpringBootBasicTag6Application.java  # Main-Klasse
│   ├── model/
│   │   └── Person.java                       # Model-Klasse
│   ├── service/
│   │   ├── PersonService.java                # Interface
│   │   ├── InMemoryPersonService.java        # Implementierung 1 (@Primary)
│   │   └── CachedPersonService.java          # Implementierung 2
│   ├── controller/
│   │   ├── HomeController.java               # Startseite
│   │   └── PersonViewController.java         # Personen-CRUD
│   ├── aspect/
│   │   ├── LoggingAspect.java                # Automatisches Logging
│   │   └── PerformanceAspect.java            # Performance-Messung
│   └── annotation/
│       └── Timed.java                        # Custom Annotation
├── src/main/resources/
│   ├── application.properties                # Konfiguration
│   ├── static/css/
│   │   └── style.css                         # Dark Style (java-developer.online)
│   └── templates/
│       ├── index.html                        # Startseite
│       ├── persons-list.html                 # Personen-Liste
│       └── person-form.html                  # Formular
└── pom.xml                                   # Maven Dependencies
```

---

## 🎨 Features im Detail

### 1. Personen-Verwaltung
- **Liste** aller Personen anzeigen
- **Erstellen** neuer Personen
- **Bearbeiten** bestehender Personen
- **Löschen** von Personen

### 2. Automatisches Logging
Jeder Service-Aufruf wird geloggt:
```
→ Calling InMemoryPersonService.getAllPersons
← Finished InMemoryPersonService.getAllPersons
```

### 3. Performance-Messung
Methoden mit `@Timed` Annotation werden gemessen:
```java
@Timed
public Person createPerson(Person person) {
    // ...
}
```

Console-Output:
```
⏱️ InMemoryPersonService.createPerson took 2ms
```

### 4. Bean Lifecycle
Beim Start:
```
🔵 InMemoryPersonService wird initialisiert...
✅ InMemoryPersonService initialisiert mit 3 Test-Personen
```

Beim Shutdown:
```
🔴 InMemoryPersonService wird heruntergefahren...
📊 Finale Anzahl Personen: 5
```

---

## 🔍 Code-Beispiele

### Constructor Injection (Best Practice)

**Mit Lombok:**
```java
@Controller
@RequiredArgsConstructor  // Constructor wird automatisch generiert!
public class PersonViewController {
    private final PersonService personService;
}
```

**Ohne Lombok:**
```java
@Controller
public class PersonViewController {
    private final PersonService personService;
    
    @Autowired  // Optional ab Spring 4.3!
    public PersonViewController(PersonService personService) {
        this.personService = personService;
    }
}
```

### Multiple Implementierungen

**InMemoryPersonService (Standard):**
```java
@Service
@Primary  // Diese Bean wird standardmäßig injiziert
public class InMemoryPersonService implements PersonService {
    // ...
}
```

**CachedPersonService (Alternative):**
```java
@Service  // Ohne @Primary - muss explizit angefordert werden
public class CachedPersonService implements PersonService {
    // ...
}
```

**Verwendung:**
```java
// Standard (InMemoryPersonService)
private final PersonService personService;

// Alternative mit @Qualifier
@Qualifier("cachedPersonService")
private final PersonService personService;
```

### AOP Logging

**LoggingAspect:**
```java
@Aspect
@Component
public class LoggingAspect {
    
    @Before("execution(* com.javafleet.springboot.service.*.*(..))")
    public void logBefore(JoinPoint joinPoint) {
        String methodName = joinPoint.getSignature().getName();
        String className = joinPoint.getTarget().getClass().getSimpleName();
        log.info("→ Calling {}.{}", className, methodName);
    }
    
    @After("execution(* com.javafleet.springboot.service.*.*(..))")
    public void logAfter(JoinPoint joinPoint) {
        String methodName = joinPoint.getSignature().getName();
        String className = joinPoint.getTarget().getClass().getSimpleName();
        log.info("← Finished {}.{}", className, methodName);
    }
}
```

### Performance-Messung mit @Around

**Custom Annotation:**
```java
@Target(ElementType.METHOD)
@Retention(RetentionPolicy.RUNTIME)
public @interface Timed {
}
```

**Aspect:**
```java
@Aspect
@Component
public class PerformanceAspect {
    
    @Around("@annotation(com.javafleet.springboot.annotation.Timed)")
    public Object measurePerformance(ProceedingJoinPoint joinPoint) throws Throwable {
        long start = System.currentTimeMillis();
        Object result = joinPoint.proceed();  // Methode ausführen
        long duration = System.currentTimeMillis() - start;
        log.info("⏱️ {} took {}ms", methodName, duration);
        return result;
    }
}
```

**Verwendung:**
```java
@Service
public class PersonService {
    
    @Timed  // Wird automatisch gemessen!
    public Person createPerson(Person person) {
        // Business-Logic
    }
}
```

---

## 🛠️ Technologie-Stack

- **Java 17**
- **Spring Boot 3.2.0**
  - spring-boot-starter-web
  - spring-boot-starter-thymeleaf
  - spring-boot-starter-aop
- **Lombok** (für @RequiredArgsConstructor)
- **Thymeleaf** (Template Engine)
- **SLF4J / Logback** (Logging)
- **Maven** (Build Tool)

---

## 📚 Dependencies (pom.xml)

```xml
<!-- Spring Boot Web -->
<dependency>
    <groupId>org.springframework.boot</groupId>
    <artifactId>spring-boot-starter-web</artifactId>
</dependency>

<!-- Thymeleaf -->
<dependency>
    <groupId>org.springframework.boot</groupId>
    <artifactId>spring-boot-starter-thymeleaf</artifactId>
</dependency>

<!-- AOP -->
<dependency>
    <groupId>org.springframework.boot</groupId>
    <artifactId>spring-boot-starter-aop</artifactId>
</dependency>

<!-- Lombok -->
<dependency>
    <groupId>org.projectlombok</groupId>
    <artifactId>lombok</artifactId>
    <optional>true</optional>
</dependency>
```

---

## 🎓 Lernziele

Nach dem Durcharbeiten dieses Projekts kannst du:

- ✅ Dependency Injection komplett verstehen
- ✅ Constructor Injection richtig einsetzen
- ✅ @Qualifier und @Primary nutzen
- ✅ Bean Lifecycle verstehen
- ✅ AOP mit @Aspect erstellen
- ✅ Custom Annotations entwickeln
- ✅ Cross-Cutting Concerns elegant lösen

---

## 📖 Weiterführende Links

- [Spring Boot Dokumentation](https://docs.spring.io/spring-boot/docs/current/reference/html/)
- [Spring AOP Dokumentation](https://docs.spring.io/spring-framework/docs/current/reference/html/core.html#aop)
- [Baeldung: Spring AOP](https://www.baeldung.com/spring-aop)
- [java-developer.online - Tag 6](https://www.java-developer.online/tag-6-di-aop-injection-aspektorientierte/)

---

## 💡 Tipps

### Console-Ausgaben beobachten
Starte die App und öffne die Console. Du siehst:
1. Bean Lifecycle (`@PostConstruct`)
2. Automatisches Logging (`LoggingAspect`)
3. Performance-Messung (`PerformanceAspect`)

### Test-Personen
Die App startet mit 3 Test-Personen:
- Elyndra Valen (elyndra@java-fleet.com)
- Nova Trent (nova@java-fleet.com)
- Franz-Martin Bernd (franz@java-fleet.com)

### Implementierung wechseln
Um `CachedPersonService` zu nutzen:

**Option 1: @Primary ändern**
```java
@Service
@Primary  // Von InMemory zu Cached verschieben
public class CachedPersonService implements PersonService {
```

**Option 2: @Qualifier nutzen**
```java
@Qualifier("cachedPersonService")
private final PersonService personService;
```

---

## 🐛 Troubleshooting

### Problem: Aspect wird nicht ausgeführt

**Lösung:** Prüfe `pom.xml`
```xml
<dependency>
    <groupId>org.springframework.boot</groupId>
    <artifactId>spring-boot-starter-aop</artifactId>
</dependency>
```

### Problem: "No qualifying bean" Fehler

**Ursache:** Mehrere Beans, kein @Primary

**Lösung 1:** @Primary hinzufügen
```java
@Service
@Primary
public class InMemoryPersonService implements PersonService {
```

**Lösung 2:** @Qualifier nutzen
```java
@Qualifier("inMemoryPersonService")
private final PersonService personService;
```

### Problem: @PostConstruct wird nicht aufgerufen

**Ursache:** Klasse ist kein Spring Bean

**Lösung:** @Service/@Component hinzufügen
```java
@Service  // Bean wird von Spring verwaltet!
public class PersonService {
    @PostConstruct
    public void init() {
        // Wird jetzt aufgerufen!
    }
}
```

---

## 📧 Kontakt

Fragen oder Feedback? Schreib mir!

**Elyndra Valen**  
Senior Entwicklerin bei Java Fleet Systems Consulting  
📧 elyndra@java-developer.online  
🌐 [java-developer.online](https://www.java-developer.online)

---

## 🔗 Java Fleet Systems Consulting

Dieses Projekt ist Teil der **Spring Boot Basic** Kurs-Serie:

- ✅ Tag 1: Erste REST API
- ✅ Tag 2: Spring Container & DI
- ✅ Tag 3: @Controller & Thymeleaf Basics
- ✅ Tag 4: Thymeleaf Forms & MVC-Pattern
- ✅ Tag 5: Konfiguration & Logging
- **✅ Tag 6: DI & AOP im Detail** ← Du bist hier!
- Tag 7: Scopes in Spring
- Tag 8: WebSockets
- Tag 9: JAX-RS in Spring Boot
- Tag 10: Integration & Abschluss

---

**Keep coding, keep learning! 💙**

*Ein Projekt von Java Fleet Systems Consulting*
