package com.javafleet.springboot.aspect;

import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.After;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

/**
 * Logging Aspect
 * 
 * Loggt automatisch alle Methoden-Aufrufe in Service-Klassen.
 * 
 * Beispiel Console-Output:
 * → Calling PersonService.createPerson
 * ← Finished PersonService.createPerson
 * 
 * @author Java Fleet Systems Consulting
 */
@Aspect
@Component
public class LoggingAspect {
    
    private static final Logger log = LoggerFactory.getLogger(LoggingAspect.class);
    
    /**
     * @Before Advice
     * Wird VOR jeder Service-Methode ausgeführt
     * 
     * Pointcut-Expression:
     * execution(* com.javafleet.springboot.service.*.*(..))
     *           ↑        ↑                             ↑  ↑
     *      Rückgabe   Package                      Klasse Methode(Parameter)
     *       (egal)                                  (alle) (alle)
     */
    @Before("execution(* com.javafleet.springboot.service.*.*(..))")
    public void logBefore(JoinPoint joinPoint) {
        String methodName = joinPoint.getSignature().getName();
        String className = joinPoint.getTarget().getClass().getSimpleName();
        
        log.info("→ Calling {}.{}", className, methodName);
    }
    
    /**
     * @After Advice
     * Wird NACH jeder Service-Methode ausgeführt (auch bei Exception!)
     */
    @After("execution(* com.javafleet.springboot.service.*.*(..))")
    public void logAfter(JoinPoint joinPoint) {
        String methodName = joinPoint.getSignature().getName();
        String className = joinPoint.getTarget().getClass().getSimpleName();
        
        log.info("← Finished {}.{}", className, methodName);
    }
}
