package com.javafleet.springboot.aspect;

import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

/**
 * Performance Aspect
 * 
 * Misst automatisch die Ausführungszeit von Methoden mit @Timed Annotation.
 * 
 * Beispiel Console-Output:
 * ⏱️ PersonService.createPerson took 3ms
 * 
 * @author Java Fleet Systems Consulting
 */
@Aspect
@Component
public class PerformanceAspect {
    
    private static final Logger log = LoggerFactory.getLogger(PerformanceAspect.class);
    
    /**
     * @Around Advice
     * Läuft VOR UND NACH der Methode
     * 
     * Pointcut: Alle Methoden mit @Timed Annotation
     * 
     * @Around ist der mächtigste Advice-Typ:
     * - Kann die Methode ausführen oder überspringen
     * - Kann Rückgabewert ändern
     * - Kann Exceptions abfangen
     */
    @Around("@annotation(com.javafleet.springboot.annotation.Timed)")
    public Object measurePerformance(ProceedingJoinPoint joinPoint) throws Throwable {
        String methodName = joinPoint.getSignature().getName();
        String className = joinPoint.getTarget().getClass().getSimpleName();
        
        // VOR der Methode - Startzeit merken
        long start = System.currentTimeMillis();
        
        Object result = null;
        
        try {
            // Die eigentliche Methode AUSFÜHREN
            result = joinPoint.proceed();
            
            // NACH der Methode (Erfolg) - Zeit berechnen
            long duration = System.currentTimeMillis() - start;
            log.info("⏱️  {}.{} took {}ms", className, methodName, duration);
            
        } catch (Exception e) {
            // NACH der Methode (Fehler) - Zeit berechnen
            long duration = System.currentTimeMillis() - start;
            log.error("❌ {}.{} failed after {}ms", className, methodName, duration);
            
            // Exception weiterwerfen!
            throw e;
        }
        
        return result;
    }
}
