package com.javafleet.springboot.controller;

import com.javafleet.springboot.model.Person;
import com.javafleet.springboot.service.PersonService;
import lombok.RequiredArgsConstructor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * Person View Controller
 * 
 * Demonstriert Constructor Injection (BEST PRACTICE!)
 * 
 * @RequiredArgsConstructor (Lombok) generiert automatisch:
 * public PersonViewController(PersonService personService) {
 *     this.personService = personService;
 * }
 * 
 * @author Java Fleet Systems Consulting
 */
@Controller
@RequestMapping("/persons")
@RequiredArgsConstructor  // Constructor Injection via Lombok!
public class PersonViewController {
    
    private static final Logger log = LoggerFactory.getLogger(PersonViewController.class);
    
    // Constructor Injection - final möglich!
    private final PersonService personService;
    
    // Spring injiziert automatisch InMemoryPersonService (@Primary)
    // Wenn du CachedPersonService willst: @Qualifier("cachedPersonService") nutzen
    
    /**
     * GET /persons
     * Zeigt alle Personen
     */
    @GetMapping
    public String showPersons(Model model) {
        log.info("Displaying all persons");
        
        List<Person> persons = personService.getAllPersons();
        model.addAttribute("persons", persons);
        model.addAttribute("title", "Personen-Übersicht");
        
        return "persons-list";
    }
    
    /**
     * GET /persons/add
     * Formular für neue Person
     */
    @GetMapping("/add")
    public String showAddForm(Model model) {
        log.info("Showing add person form");
        
        model.addAttribute("person", new Person());
        model.addAttribute("title", "Person hinzufügen");
        model.addAttribute("action", "/persons/add");
        
        return "person-form";
    }
    
    /**
     * POST /persons/add
     * Person erstellen
     */
    @PostMapping("/add")
    public String addPerson(@ModelAttribute Person person) {
        log.info("Adding new person: {} {}", person.getFirstname(), person.getLastname());
        
        personService.createPerson(person);
        
        return "redirect:/persons";
    }
    
    /**
     * GET /persons/edit/{id}
     * Formular zum Bearbeiten
     */
    @GetMapping("/edit/{id}")
    public String showEditForm(@PathVariable Long id, Model model) {
        log.info("Showing edit form for person ID: {}", id);
        
        Person person = personService.getPersonById(id);
        
        if (person == null) {
            log.error("Person with ID {} not found", id);
            return "redirect:/persons";
        }
        
        model.addAttribute("person", person);
        model.addAttribute("title", "Person bearbeiten");
        model.addAttribute("action", "/persons/edit");
        
        return "person-form";
    }
    
    /**
     * POST /persons/edit
     * Person aktualisieren
     */
    @PostMapping("/edit")
    public String editPerson(@ModelAttribute Person person) {
        log.info("Updating person ID: {}", person.getId());
        
        personService.updatePerson(person);
        
        return "redirect:/persons";
    }
    
    /**
     * GET /persons/delete/{id}
     * Person löschen
     */
    @GetMapping("/delete/{id}")
    public String deletePerson(@PathVariable Long id) {
        log.info("Deleting person ID: {}", id);
        
        personService.deletePerson(id);
        
        return "redirect:/persons";
    }
}
