package com.javafleet.springboot.service;

import com.javafleet.springboot.model.Person;
import jakarta.annotation.PostConstruct;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;

/**
 * Cached-Implementierung des PersonService
 * 
 * Nutzt einen HashMap-Cache für schnelleren Zugriff.
 * Beispiel für alternative Implementierung!
 * 
 * NICHT @Primary - muss mit @Qualifier explizit angefordert werden
 */
@Service  // Ohne @Primary!
public class CachedPersonService implements PersonService {
    
    private static final Logger log = LoggerFactory.getLogger(CachedPersonService.class);
    
    private final Map<Long, Person> cache = new HashMap<>();
    private final AtomicLong idCounter = new AtomicLong(1);
    
    @PostConstruct
    public void init() {
        log.info("🟡 CachedPersonService wird initialisiert...");
        log.info("💾 Cache-basierte Implementierung bereit!");
    }
    
    @Override
    public List<Person> getAllPersons() {
        log.debug("Getting all persons from cache, current count: {}", cache.size());
        return new ArrayList<>(cache.values());
    }
    
    @Override
    public Person getPersonById(Long id) {
        log.debug("Cache lookup for person ID: {}", id);
        return cache.get(id);
    }
    
    @Override
    public Person createPerson(Person person) {
        log.info("Creating person in cache: {} {}", person.getFirstname(), person.getLastname());
        
        person.setId(idCounter.getAndIncrement());
        cache.put(person.getId(), person);
        
        log.info("Person cached with ID: {}", person.getId());
        return person;
    }
    
    @Override
    public Person updatePerson(Person person) {
        log.info("Updating person in cache: ID {}", person.getId());
        
        if (cache.containsKey(person.getId())) {
            cache.put(person.getId(), person);
            log.info("Person {} updated in cache", person.getId());
            return person;
        }
        
        log.error("Person {} not found in cache!", person.getId());
        return null;
    }
    
    @Override
    public void deletePerson(Long id) {
        log.info("Deleting person from cache: ID {}", id);
        
        Person removed = cache.remove(id);
        
        if (removed != null) {
            log.info("Person {} removed from cache", id);
        } else {
            log.warn("Person {} not found in cache", id);
        }
    }
}
