package com.javafleet.springboot.service;

import com.javafleet.springboot.annotation.Timed;
import com.javafleet.springboot.model.Person;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.PreDestroy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;

/**
 * InMemory-Implementierung des PersonService
 * 
 * Speichert Personen im Arbeitsspeicher (verloren nach Neustart).
 * Perfekt für Entwicklung und Tests!
 * 
 * @Primary = Diese Implementierung ist die Standard-Wahl
 */
@Service
@Primary  // Diese Bean wird standardmäßig injiziert
public class InMemoryPersonService implements PersonService {
    
    private static final Logger log = LoggerFactory.getLogger(InMemoryPersonService.class);
    
    private final List<Person> persons = new ArrayList<>();
    private final AtomicLong idCounter = new AtomicLong(1);
    
    /**
     * @PostConstruct wird NACH Dependency Injection aufgerufen
     * Perfekt für Initialisierung!
     */
    @PostConstruct
    public void init() {
        log.info("🔵 InMemoryPersonService wird initialisiert...");
        
        // Test-Daten erstellen
        persons.add(new Person(idCounter.getAndIncrement(), "Elyndra", "Valen", "elyndra@java-fleet.com"));
        persons.add(new Person(idCounter.getAndIncrement(), "Nova", "Trent", "nova@java-fleet.com"));
        persons.add(new Person(idCounter.getAndIncrement(), "Franz-Martin", "Bernd", "franz@java-fleet.com"));
        
        log.info("✅ InMemoryPersonService initialisiert mit {} Test-Personen", persons.size());
    }
    
    /**
     * @PreDestroy wird VOR dem Shutdown aufgerufen
     * Perfekt für Cleanup!
     */
    @PreDestroy
    public void cleanup() {
        log.info("🔴 InMemoryPersonService wird heruntergefahren...");
        log.info("📊 Finale Anzahl Personen: {}", persons.size());
    }
    
    @Override
    public List<Person> getAllPersons() {
        log.debug("Getting all persons, current count: {}", persons.size());
        return new ArrayList<>(persons);
    }
    
    @Override
    public Person getPersonById(Long id) {
        log.debug("Looking for person with ID: {}", id);
        return persons.stream()
                .filter(p -> p.getId().equals(id))
                .findFirst()
                .orElse(null);
    }
    
    @Override
    @Timed  // Custom Annotation - Performance-Messung!
    public Person createPerson(Person person) {
        log.info("Creating new person: {} {}", person.getFirstname(), person.getLastname());
        
        person.setId(idCounter.getAndIncrement());
        persons.add(person);
        
        log.info("Person created successfully with ID: {}", person.getId());
        return person;
    }
    
    @Override
    @Timed  // Custom Annotation - Performance-Messung!
    public Person updatePerson(Person person) {
        log.info("Updating person with ID: {}", person.getId());
        
        for (int i = 0; i < persons.size(); i++) {
            if (persons.get(i).getId().equals(person.getId())) {
                persons.set(i, person);
                log.info("Person {} updated successfully", person.getId());
                return person;
            }
        }
        
        log.error("Failed to update person {} - not found!", person.getId());
        return null;
    }
    
    @Override
    @Timed  // Custom Annotation - Performance-Messung!
    public void deletePerson(Long id) {
        log.info("Deleting person with ID: {}", id);
        
        boolean removed = persons.removeIf(p -> p.getId().equals(id));
        
        if (removed) {
            log.info("Person {} deleted successfully", id);
        } else {
            log.warn("Could not delete person {} - not found", id);
        }
    }
}
