package online.javadeveloper.devopsadmin.commands;

import online.javadeveloper.devopsadmin.config.EnvironmentHolder;
import online.javadeveloper.devopsadmin.service.DeploymentService;
import online.javadeveloper.devopsadmin.service.ServiceRegistry;
import org.springframework.shell.Availability;
import org.springframework.shell.standard.ShellComponent;
import org.springframework.shell.standard.ShellMethod;
import org.springframework.shell.standard.ShellMethodAvailability;
import org.springframework.shell.standard.ShellOption;

/**
 * Commands für Deployments und Rollbacks.
 */
@ShellComponent
public class DeployCommands {

    private final ServiceRegistry registry;
    private final DeploymentService deploymentService;
    private final EnvironmentHolder environmentHolder;

    public DeployCommands(ServiceRegistry registry,
                          DeploymentService deploymentService,
                          EnvironmentHolder environmentHolder) {
        this.registry = registry;
        this.deploymentService = deploymentService;
        this.environmentHolder = environmentHolder;
    }

    @ShellMethod(value = "Deployed einen Service", key = "deploy")
    public String deploy(
            @ShellOption(help = "Name des Services")
            String service,

            @ShellOption(defaultValue = "latest", help = "Version/Tag")
            String version,

            @ShellOption(defaultValue = ShellOption.NULL, help = "Ziel-Environment (default: aktuelles)")
            String environment) {

        // Validierung
        if (!registry.exists(service)) {
            return "❌ Service nicht gefunden: " + service + "\n" +
                   "   Verfügbare Services: " + String.join(", ", registry.getAllServiceNames());
        }

        // Environment bestimmen
        var targetEnv = environment != null ? environment : environmentHolder.getCurrent();

        // Warnung für Production
        if ("production".equals(targetEnv)) {
            return "⚠️ Production-Deployment benötigt Bestätigung!\n" +
                   "   Nutze: deploy " + service + " --version " + version + 
                   " --environment production --confirm";
        }

        return deploymentService.deploy(service, version, targetEnv);
    }

    @ShellMethod(value = "Deployed in Production (mit Bestätigung)", key = "deploy-prod")
    public String deployProduction(
            @ShellOption(help = "Name des Services")
            String service,

            @ShellOption(help = "Version/Tag")
            String version,

            @ShellOption(defaultValue = "false", help = "Deployment bestätigen")
            boolean confirm) {

        if (!registry.exists(service)) {
            return "❌ Service nicht gefunden: " + service;
        }

        if (!confirm) {
            return """
                ⚠️ ACHTUNG: Production-Deployment!
                
                Du bist dabei, %s:%s nach PRODUCTION zu deployen.
                
                Dies betrifft echte Benutzer!
                
                Zum Bestätigen: deploy-prod %s --version %s --confirm
                """.formatted(service, version, service, version);
        }

        return deploymentService.deploy(service, version, "production");
    }

    @ShellMethod(value = "Rollback zu vorheriger Version", key = "rollback")
    public String rollback(
            @ShellOption(help = "Name des Services")
            String service,

            @ShellOption(defaultValue = ShellOption.NULL, help = "Environment (default: aktuelles)")
            String environment) {

        if (!registry.exists(service)) {
            return "❌ Service nicht gefunden: " + service;
        }

        var targetEnv = environment != null ? environment : environmentHolder.getCurrent();
        return deploymentService.rollback(service, targetEnv);
    }

    @ShellMethod(value = "Zeigt Deployment-Verlauf", key = "deploy-history")
    public String deployHistory(
            @ShellOption(defaultValue = ShellOption.NULL, help = "Filter nach Service")
            String service,

            @ShellOption(defaultValue = "10", help = "Anzahl Einträge")
            int limit) {

        return deploymentService.getHistory(service, limit);
    }

    // =====================================================================
    // Gefährliche Commands mit Availability-Check
    // =====================================================================

    @ShellMethod(value = "Löscht alle Deployment-Daten (nur Dev!)", key = "nuke-history")
    @ShellMethodAvailability("nukeAvailability")
    public String nukeHistory() {
        return "💥 Deployment-History gelöscht!";
    }

    public Availability nukeAvailability() {
        if ("production".equals(environmentHolder.getCurrent())) {
            return Availability.unavailable("Nicht in Production verfügbar!");
        }
        return Availability.available();
    }
}
