package online.javadeveloper.devopsadmin.commands;

import online.javadeveloper.devopsadmin.config.EnvironmentHolder;
import org.springframework.shell.standard.ShellComponent;
import org.springframework.shell.standard.ShellMethod;
import org.springframework.shell.standard.ShellOption;

/**
 * Commands für Environment-Verwaltung.
 */
@ShellComponent
public class EnvironmentCommands {

    private final EnvironmentHolder environmentHolder;

    public EnvironmentCommands(EnvironmentHolder environmentHolder) {
        this.environmentHolder = environmentHolder;
    }

    @ShellMethod(value = "Wechselt das aktive Environment", key = "use-env")
    public String useEnvironment(
            @ShellOption(help = "dev, staging oder production")
            String environment) {

        if (environmentHolder.setCurrent(environment)) {
            String warning = "";
            if ("production".equals(environment)) {
                warning = "\n\n⚠️  ACHTUNG: Du arbeitest jetzt in PRODUCTION!";
            }
            return "✅ Environment gewechselt zu: " + environment + warning;
        }

        return "❌ Ungültiges Environment: " + environment + "\n" +
               "   Erlaubt: " + String.join(", ", environmentHolder.getValidEnvironments());
    }

    @ShellMethod(value = "Zeigt das aktuelle Environment", key = "current-env")
    public String currentEnvironment() {
        var env = environmentHolder.getCurrent();
        var icon = switch (env) {
            case "production" -> "🔴";
            case "staging" -> "🟡";
            case "dev" -> "🟢";
            default -> "⚪";
        };

        return icon + " Aktuelles Environment: " + env;
    }

    @ShellMethod(value = "Listet alle verfügbaren Environments", key = "list-envs")
    public String listEnvironments() {
        var current = environmentHolder.getCurrent();
        var sb = new StringBuilder();
        sb.append("🌍 Verfügbare Environments:\n\n");

        for (var env : environmentHolder.getValidEnvironments()) {
            var marker = env.equals(current) ? " ← aktuell" : "";
            var icon = switch (env) {
                case "production" -> "🔴";
                case "staging" -> "🟡";
                case "dev" -> "🟢";
                default -> "⚪";
            };
            sb.append("  ").append(icon).append(" ").append(env).append(marker).append("\n");
        }

        return sb.toString();
    }
}
