package online.javadeveloper.devopsadmin.commands;

import online.javadeveloper.devopsadmin.service.HealthChecker;
import online.javadeveloper.devopsadmin.service.ServiceRegistry;
import org.springframework.shell.standard.ShellComponent;
import org.springframework.shell.standard.ShellMethod;
import org.springframework.shell.standard.ShellOption;

/**
 * Commands für Service-Verwaltung und Health Checks.
 */
@ShellComponent
public class ServiceCommands {

    private final ServiceRegistry registry;
    private final HealthChecker healthChecker;

    public ServiceCommands(ServiceRegistry registry, HealthChecker healthChecker) {
        this.registry = registry;
        this.healthChecker = healthChecker;
    }

    @ShellMethod(value = "Listet alle registrierten Services", key = "services")
    public String listServices() {
        var services = registry.getAllServices();

        if (services.isEmpty()) {
            return "⚠️ Keine Services registriert.";
        }

        var sb = new StringBuilder();
        sb.append("📋 Registrierte Services:\n\n");

        sb.append(String.format("  %-25s %-35s %s%n", "NAME", "URL", "VERSION"));
        sb.append("  ").append("─".repeat(70)).append("\n");

        for (var service : services) {
            sb.append(String.format("  %-25s %-35s %s%n",
                    service.name(),
                    service.url(),
                    service.currentVersion()));
        }

        sb.append("\n");
        sb.append("Gesamt: ").append(services.size()).append(" Services\n");

        return sb.toString();
    }

    @ShellMethod(value = "Prüft Health eines oder aller Services", key = "health-check")
    public String healthCheck(
            @ShellOption(
                    defaultValue = ShellOption.NULL,
                    help = "Service-Name (leer = alle Services prüfen)"
            ) String service) {

        if (service == null) {
            return healthChecker.checkAll();
        }

        return healthChecker.check(service);
    }

    @ShellMethod(value = "Zeigt Logs eines Services", key = "logs")
    public String logs(
            @ShellOption(help = "Name des Services")
            String service,

            @ShellOption(defaultValue = "20", help = "Anzahl Zeilen (1-100)")
            int lines,

            @ShellOption(defaultValue = "false", help = "Nur Errors anzeigen")
            boolean errorsOnly) {

        // Validierung
        if (lines < 1 || lines > 100) {
            return "❌ --lines muss zwischen 1 und 100 liegen.";
        }

        if (!registry.exists(service)) {
            return "❌ Service nicht gefunden: " + service + "\n" +
                   "   Verfügbare Services: " + String.join(", ", registry.getAllServiceNames());
        }

        return registry.getLogs(service, lines, errorsOnly);
    }

    @ShellMethod(value = "Zeigt Details zu einem Service", key = "service-info")
    public String serviceInfo(
            @ShellOption(help = "Name des Services")
            String service) {

        return registry.get(service)
                .map(info -> {
                    var sb = new StringBuilder();
                    sb.append("📦 Service-Details: ").append(info.name()).append("\n\n");
                    sb.append("  Name:        ").append(info.name()).append("\n");
                    sb.append("  URL:         ").append(info.url()).append("\n");
                    sb.append("  Version:     ").append(info.currentVersion()).append("\n");
                    sb.append("  Environment: ").append(info.environment()).append("\n");
                    return sb.toString();
                })
                .orElse("❌ Service nicht gefunden: " + service);
    }
}
